/***************************************************************************
 # Copyright (c) 2019, NVIDIA CORPORATION.  All rights reserved.
 #
 # NVIDIA CORPORATION and its licensors retain all intellectual property
 # and proprietary rights in and to this software, related documentation
 # and any modifications thereto.  Any use, reproduction, disclosure or
 # distribution of this software and related documentation without an express
 # license agreement from NVIDIA CORPORATION is strictly prohibited.
 **************************************************************************/
#pragma once
#include "GBuffer.h"
#include "PrtParams.h"
#include "Falcor.h"
#include "FalcorExperimental.h"
#include "Core/Framework.h"
#include "RenderGraph/BasePasses/ComputePass.h"
#include "Utils/Sampling/SampleGenerator.h"
#include "RenderGraph/RenderPassHelpers.h"

using namespace Falcor;

/** Base class for PRT
*/
class PrtBase : public GBuffer, inherit_shared_from_this<GBuffer, PrtBase>
{
public:
    using SharedPtr = std::shared_ptr<PrtBase>;

    virtual Dictionary getScriptingDictionary() override;
    RenderPassReflection reflect(const CompileData& compileData) override;
    virtual void execute(RenderContext* pRenderContext, const RenderData& renderData) override;
    virtual void setScene(RenderContext* pRenderContext, const Scene::SharedPtr& pScene) override;  
    virtual void compile(RenderContext* pContext, const CompileData& compileData) override;

    virtual bool onKeyEvent(const KeyboardEvent& keyEvent) override;
    virtual void renderUI(Gui::Widgets& widget) override;

protected:
    PrtBase();
    virtual void setCullMode(RasterizerState::CullMode mode) override;
    virtual bool parseDictionary(const Dictionary& dict) override;

    virtual void preparePRT();
    virtual void prepareShaders();
    virtual void preprocessScene();

    void precomputeSH(RenderContext* pRenderContext, const RenderData& renderData);
    void interpolateHermite(RenderContext* pRenderContext, const RenderData& renderData);
    void interpolateTrilinear(RenderContext* pRenderContext, const RenderData& renderData);
    void interpolateTaylor(RenderContext* pRenderContext, const RenderData& renderData);

    virtual void computePerVertex(RenderContext* pRenderContext, const RenderData& renderData);
    virtual void computePerVertexPerLight(RenderContext* pRenderContext, const RenderData& renderData);

    // Animation
    void prepAnimation();
    void calcLightCenters(RenderContext* pRenderContext, const RenderData& renderData);
    void setObjToWorldMatrices(std::vector<float4x4> *before = NULL, std::vector<float4x4> *after = NULL);

    // 0: None, 1: Dragon and bunny
    void animateDragonAndBunny();

    bool mLightCentersReady = false;
    ComputePass::SharedPtr mpLightCenterPass;
    TypedBufferBase::SharedPtr mpLightCenters;
    std::vector<float4x4> mObjectToWorld;

    bool mPauseAnimation = false;
    bool mPrintDebug = false;

    // Internal state
    RenderGraph::SharedPtr          mpDepthPrePassGraph;
    Fbo::SharedPtr                  mpFbo;

    // Rasterization resources
    struct
    {
        GraphicsState::SharedPtr pState;
        GraphicsProgram::SharedPtr pProgram;
        GraphicsVars::SharedPtr pVars;
    } mRaster;

    // UI variables
    bool mOptionsChanged = false;
    // 0: Analytic 2018; 1: Ours Hermite; 2: Ours trilinear
    // 3: Taylor-series; 4: Analytic 2018 (more robust to multi-lights)
    uint32_t mAlgo = 1;
    uint32_t mAnimation = 0;

    std::string mOutDir = "";
    std::string mCoeffFilename = "";
    uint32_t mGridReso = 8;
    float mLightDisplayScale = 1.f;
    float mLightIntensityScale = 1.f;

    // PRT viewer data
    SampleGenerator::SharedPtr mpSampleGenerator;

    PrtParams mPrtParams;
    TypedBufferBase::SharedPtr mpTcoeffs;

    BoundingBox mBoundExLight;
    std::vector<uint2> mEmitterToMeshData;
    TypedBufferBase::SharedPtr mpEmitterToMesh;

    Texture::SharedPtr mpLegendre2345, mpLegendre6789;
    Sampler::SharedPtr mpLegendreSampler;

    ComputePass::SharedPtr mpComputePerVertexPass;
    TypedBufferBase::SharedPtr mpVertexColors;

    ComputePass::SharedPtr mpComputePerVertexPerLightPass;

    ComputePass::SharedPtr mpPrecomputePass;
    TypedBufferBase::SharedPtr mpZhGrad;

    ComputePass::SharedPtr mpZhToShPass;
    TypedBufferBase::SharedPtr mpShGrad[3];

    ComputePass::SharedPtr mpHermitePass;

    ComputePass::SharedPtr mpTrilinearPass;
    ComputePass::SharedPtr mpTaylorPass;

    std::string mShaderDir = "";
    std::string mSMVersion = "5_1";
};
