/*
    This file is part of Mitsuba, a physically based rendering system.

    Copyright (c) 2007-2014 by Wenzel Jakob and others.

    Mitsuba is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License Version 3
    as published by the Free Software Foundation.

    Mitsuba is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see <http://www.gnu.org/licenses/>.
*/

#include <mitsuba/core/qmc.h>

MTS_NAMESPACE_BEGIN

const int primeTable[primeTableSize] = {
    2, 3, 5, 7, 11, 13, 17, 19, 23, 29, 31, 37, 41, 43, 47, 53, 59, 61, 67, 71, 73, 79, 83, 89, 97, 101, 103, 107,
    109, 113, 127, 131, 137, 139, 149, 151, 157, 163, 167, 173, 179, 181, 191, 193, 197, 199, 211, 223, 227,
    229, 233, 239, 241, 251, 257, 263, 269, 271, 277, 281, 283, 293, 307, 311, 313, 317, 331, 337, 347, 349,
    353, 359, 367, 373, 379, 383, 389, 397, 401, 409, 419, 421, 431, 433, 439, 443, 449, 457, 461, 463, 467,
    479, 487, 491, 499, 503, 509, 521, 523, 541, 547, 557, 563, 569, 571, 577, 587, 593, 599, 601, 607, 613,
    617, 619, 631, 641, 643, 647, 653, 659, 661, 673, 677, 683, 691, 701, 709, 719, 727, 733, 739, 743, 751,
    757, 761, 769, 773, 787, 797, 809, 811, 821, 823, 827, 829, 839, 853, 857, 859, 863, 877, 881, 883, 887,
    907, 911, 919, 929, 937, 941, 947, 953, 967, 971, 977, 983, 991, 997, 1009, 1013, 1019, 1021, 1031, 1033,
    1039, 1049, 1051, 1061, 1063, 1069, 1087, 1091, 1093, 1097, 1103, 1109, 1117, 1123, 1129, 1151, 1153,
    1163, 1171, 1181, 1187, 1193, 1201, 1213, 1217, 1223, 1229, 1231, 1237, 1249, 1259, 1277, 1279, 1283,
    1289, 1291, 1297, 1301, 1303, 1307, 1319, 1321, 1327, 1361, 1367, 1373, 1381, 1399, 1409, 1423, 1427,
    1429, 1433, 1439, 1447, 1451, 1453, 1459, 1471, 1481, 1483, 1487, 1489, 1493, 1499, 1511, 1523, 1531,
    1543, 1549, 1553, 1559, 1567, 1571, 1579, 1583, 1597, 1601, 1607, 1609, 1613, 1619, 1621, 1627, 1637,
    1657, 1663, 1667, 1669, 1693, 1697, 1699, 1709, 1721, 1723, 1733, 1741, 1747, 1753, 1759, 1777, 1783,
    1787, 1789, 1801, 1811, 1823, 1831, 1847, 1861, 1867, 1871, 1873, 1877, 1879, 1889, 1901, 1907, 1913,
    1931, 1933, 1949, 1951, 1973, 1979, 1987, 1993, 1997, 1999, 2003, 2011, 2017, 2027, 2029, 2039, 2053,
    2063, 2069, 2081, 2083, 2087, 2089, 2099, 2111, 2113, 2129, 2131, 2137, 2141, 2143, 2153, 2161, 2179,
    2203, 2207, 2213, 2221, 2237, 2239, 2243, 2251, 2267, 2269, 2273, 2281, 2287, 2293, 2297, 2309, 2311,
    2333, 2339, 2341, 2347, 2351, 2357, 2371, 2377, 2381, 2383, 2389, 2393, 2399, 2411, 2417, 2423, 2437,
    2441, 2447, 2459, 2467, 2473, 2477, 2503, 2521, 2531, 2539, 2543, 2549, 2551, 2557, 2579, 2591, 2593,
    2609, 2617, 2621, 2633, 2647, 2657, 2659, 2663, 2671, 2677, 2683, 2687, 2689, 2693, 2699, 2707, 2711,
    2713, 2719, 2729, 2731, 2741, 2749, 2753, 2767, 2777, 2789, 2791, 2797, 2801, 2803, 2819, 2833, 2837,
    2843, 2851, 2857, 2861, 2879, 2887, 2897, 2903, 2909, 2917, 2927, 2939, 2953, 2957, 2963, 2969, 2971,
    2999, 3001, 3011, 3019, 3023, 3037, 3041, 3049, 3061, 3067, 3079, 3083, 3089, 3109, 3119, 3121, 3137,
    3163, 3167, 3169, 3181, 3187, 3191, 3203, 3209, 3217, 3221, 3229, 3251, 3253, 3257, 3259, 3271, 3299,
    3301, 3307, 3313, 3319, 3323, 3329, 3331, 3343, 3347, 3359, 3361, 3371, 3373, 3389, 3391, 3407, 3413,
    3433, 3449, 3457, 3461, 3463, 3467, 3469, 3491, 3499, 3511, 3517, 3527, 3529, 3533, 3539, 3541, 3547,
    3557, 3559, 3571, 3581, 3583, 3593, 3607, 3613, 3617, 3623, 3631, 3637, 3643, 3659, 3671, 3673, 3677,
    3691, 3697, 3701, 3709, 3719, 3727, 3733, 3739, 3761, 3767, 3769, 3779, 3793, 3797, 3803, 3821, 3823,
    3833, 3847, 3851, 3853, 3863, 3877, 3881, 3889, 3907, 3911, 3917, 3919, 3923, 3929, 3931, 3943, 3947,
    3967, 3989, 4001, 4003, 4007, 4013, 4019, 4021, 4027, 4049, 4051, 4057, 4073, 4079, 4091, 4093, 4099,
    4111, 4127, 4129, 4133, 4139, 4153, 4157, 4159, 4177, 4201, 4211, 4217, 4219, 4229, 4231, 4241, 4243,
    4253, 4259, 4261, 4271, 4273, 4283, 4289, 4297, 4327, 4337, 4339, 4349, 4357, 4363, 4373, 4391, 4397,
    4409, 4421, 4423, 4441, 4447, 4451, 4457, 4463, 4481, 4483, 4493, 4507, 4513, 4517, 4519, 4523, 4547,
    4549, 4561, 4567, 4583, 4591, 4597, 4603, 4621, 4637, 4639, 4643, 4649, 4651, 4657, 4663, 4673, 4679,
    4691, 4703, 4721, 4723, 4729, 4733, 4751, 4759, 4783, 4787, 4789, 4793, 4799, 4801, 4813, 4817, 4831,
    4861, 4871, 4877, 4889, 4903, 4909, 4919, 4931, 4933, 4937, 4943, 4951, 4957, 4967, 4969, 4973, 4987,
    4993, 4999, 5003, 5009, 5011, 5021, 5023, 5039, 5051, 5059, 5077, 5081, 5087, 5099, 5101, 5107, 5113,
    5119, 5147, 5153, 5167, 5171, 5179, 5189, 5197, 5209, 5227, 5231, 5233, 5237, 5261, 5273, 5279, 5281,
    5297, 5303, 5309, 5323, 5333, 5347, 5351, 5381, 5387, 5393, 5399, 5407, 5413, 5417, 5419, 5431, 5437,
    5441, 5443, 5449, 5471, 5477, 5479, 5483, 5501, 5503, 5507, 5519, 5521, 5527, 5531, 5557, 5563, 5569,
    5573, 5581, 5591, 5623, 5639, 5641, 5647, 5651, 5653, 5657, 5659, 5669, 5683, 5689, 5693, 5701, 5711,
    5717, 5737, 5741, 5743, 5749, 5779, 5783, 5791, 5801, 5807, 5813, 5821, 5827, 5839, 5843, 5849, 5851,
    5857, 5861, 5867, 5869, 5879, 5881, 5897, 5903, 5923, 5927, 5939, 5953, 5981, 5987, 6007, 6011, 6029,
    6037, 6043, 6047, 6053, 6067, 6073, 6079, 6089, 6091, 6101, 6113, 6121, 6131, 6133, 6143, 6151, 6163,
    6173, 6197, 6199, 6203, 6211, 6217, 6221, 6229, 6247, 6257, 6263, 6269, 6271, 6277, 6287, 6299, 6301,
    6311, 6317, 6323, 6329, 6337, 6343, 6353, 6359, 6361, 6367, 6373, 6379, 6389, 6397, 6421, 6427, 6449,
    6451, 6469, 6473, 6481, 6491, 6521, 6529, 6547, 6551, 6553, 6563, 6569, 6571, 6577, 6581, 6599, 6607,
    6619, 6637, 6653, 6659, 6661, 6673, 6679, 6689, 6691, 6701, 6703, 6709, 6719, 6733, 6737, 6761, 6763,
    6779, 6781, 6791, 6793, 6803, 6823, 6827, 6829, 6833, 6841, 6857, 6863, 6869, 6871, 6883, 6899, 6907,
    6911, 6917, 6947, 6949, 6959, 6961, 6967, 6971, 6977, 6983, 6991, 6997, 7001, 7013, 7019, 7027, 7039,
    7043, 7057, 7069, 7079, 7103, 7109, 7121, 7127, 7129, 7151, 7159, 7177, 7187, 7193, 7207, 7211, 7213,
    7219, 7229, 7237, 7243, 7247, 7253, 7283, 7297, 7307, 7309, 7321, 7331, 7333, 7349, 7351, 7369, 7393,
    7411, 7417, 7433, 7451, 7457, 7459, 7477, 7481, 7487, 7489, 7499, 7507, 7517, 7523, 7529, 7537, 7541,
    7547, 7549, 7559, 7561, 7573, 7577, 7583, 7589, 7591, 7603, 7607, 7621, 7639, 7643, 7649, 7669, 7673,
    7681, 7687, 7691, 7699, 7703, 7717, 7723, 7727, 7741, 7753, 7757, 7759, 7789, 7793, 7817, 7823, 7829,
    7841, 7853, 7867, 7873, 7877, 7879, 7883, 7901, 7907, 7919, 7927, 7933, 7937, 7949, 7951, 7963, 7993,
    8009, 8011, 8017, 8039, 8053, 8059, 8069, 8081, 8087, 8089, 8093, 8101, 8111, 8117, 8123, 8147, 8161
};

/* Radical inverse: generic versions */

Float radicalInverse(int base, uint64_t i) {
    Float radical = (Float) 1 / (Float) base;
    Float inverse = 0.0f, digit = radical;

    while (i) {
        inverse += digit * (Float) (i % base);
        digit *= radical;
        i /= base;
    }

    return std::min(inverse, ONE_MINUS_EPS);
}

Float scrambledRadicalInverse(int base, uint64_t i, uint16_t *perm) {
    Float radical = (Float) 1 / (Float) base;
    Float inverse = 0.0f, digit = radical;

    while (i) {
        inverse += digit * (Float) perm[i % base];
        digit *= radical;
        i /= base;
    }

    inverse += digit * perm[0] / (1 - radical);

    return std::min(inverse, ONE_MINUS_EPS);
}

Float radicalInverseIncremental(int base, Float x) {
    Float radical = (Float) 1 / (Float) base;
    Float h, hh, r = 1.0f - x - (Float) 1e-10;

    if (radical < r) {
        x += radical;
    } else {
        h = radical;
        do {
            hh = h;
            h *= radical;
        } while (h >= r);

        x += hh + h - 1.0f;
    }

    return x;
}

/* Radical inverse: explicitly instantiated versions that permit important
   additional compiler optimizations.

   The code is designed to keep FP and integer math separate in the loop (to
   avoid introducing unnecessary store-to-load dependencies on current
   processor architectures) */

#define RINV(base) { \
        const Float radical = (Float) 1 / (Float) base; \
        uint64_t value = 0; \
        Float factor = 1.0f; \
        while (index) { \
            uint64_t next  = index / base; \
            uint64_t digit = index - next*base; \
            value = value * base + digit; \
            factor *= radical; \
            index = next; \
        } \
        inverse = (Float) value * factor; \
    }


#define SCRAMBLED_RINV(base) { \
        const Float radical = (Float) 1 / (Float) base; \
        uint64_t value = 0; \
        Float factor = 1.0f; \
        while (index) { \
            uint64_t next  = index / base; \
            uint64_t digit = index - next*base; \
            value = value * base + perm[digit]; \
            factor *= radical; \
            index = next; \
        } \
        inverse = factor * ((Float) value + radical * perm[0] / (1 - radical)); \
    }

Float radicalInverseFast(uint16_t baseIndex, uint64_t index) {
    Float inverse = 0.0f;

    switch (baseIndex) {
        case 0: RINV(2); break;
        case 1: RINV(3); break;
        case 2: RINV(5); break;
        case 3: RINV(7); break;
        case 4: RINV(11); break;
        case 5: RINV(13); break;
        case 6: RINV(17); break;
        case 7: RINV(19); break;
        case 8: RINV(23); break;
        case 9: RINV(29); break;
        case 10: RINV(31); break;
        case 11: RINV(37); break;
        case 12: RINV(41); break;
        case 13: RINV(43); break;
        case 14: RINV(47); break;
        case 15: RINV(53); break;
        case 16: RINV(59); break;
        case 17: RINV(61); break;
        case 18: RINV(67); break;
        case 19: RINV(71); break;
        case 20: RINV(73); break;
        case 21: RINV(79); break;
        case 22: RINV(83); break;
        case 23: RINV(89); break;
        case 24: RINV(97); break;
        case 25: RINV(101); break;
        case 26: RINV(103); break;
        case 27: RINV(107); break;
        case 28: RINV(109); break;
        case 29: RINV(113); break;
        case 30: RINV(127); break;
        case 31: RINV(131); break;
        case 32: RINV(137); break;
        case 33: RINV(139); break;
        case 34: RINV(149); break;
        case 35: RINV(151); break;
        case 36: RINV(157); break;
        case 37: RINV(163); break;
        case 38: RINV(167); break;
        case 39: RINV(173); break;
        case 40: RINV(179); break;
        case 41: RINV(181); break;
        case 42: RINV(191); break;
        case 43: RINV(193); break;
        case 44: RINV(197); break;
        case 45: RINV(199); break;
        case 46: RINV(211); break;
        case 47: RINV(223); break;
        case 48: RINV(227); break;
        case 49: RINV(229); break;
        case 50: RINV(233); break;
        case 51: RINV(239); break;
        case 52: RINV(241); break;
        case 53: RINV(251); break;
        case 54: RINV(257); break;
        case 55: RINV(263); break;
        case 56: RINV(269); break;
        case 57: RINV(271); break;
        case 58: RINV(277); break;
        case 59: RINV(281); break;
        case 60: RINV(283); break;
        case 61: RINV(293); break;
        case 62: RINV(307); break;
        case 63: RINV(311); break;
        case 64: RINV(313); break;
        case 65: RINV(317); break;
        case 66: RINV(331); break;
        case 67: RINV(337); break;
        case 68: RINV(347); break;
        case 69: RINV(349); break;
        case 70: RINV(353); break;
        case 71: RINV(359); break;
        case 72: RINV(367); break;
        case 73: RINV(373); break;
        case 74: RINV(379); break;
        case 75: RINV(383); break;
        case 76: RINV(389); break;
        case 77: RINV(397); break;
        case 78: RINV(401); break;
        case 79: RINV(409); break;
        case 80: RINV(419); break;
        case 81: RINV(421); break;
        case 82: RINV(431); break;
        case 83: RINV(433); break;
        case 84: RINV(439); break;
        case 85: RINV(443); break;
        case 86: RINV(449); break;
        case 87: RINV(457); break;
        case 88: RINV(461); break;
        case 89: RINV(463); break;
        case 90: RINV(467); break;
        case 91: RINV(479); break;
        case 92: RINV(487); break;
        case 93: RINV(491); break;
        case 94: RINV(499); break;
        case 95: RINV(503); break;
        case 96: RINV(509); break;
        case 97: RINV(521); break;
        case 98: RINV(523); break;
        case 99: RINV(541); break;
        case 100: RINV(547); break;
        case 101: RINV(557); break;
        case 102: RINV(563); break;
        case 103: RINV(569); break;
        case 104: RINV(571); break;
        case 105: RINV(577); break;
        case 106: RINV(587); break;
        case 107: RINV(593); break;
        case 108: RINV(599); break;
        case 109: RINV(601); break;
        case 110: RINV(607); break;
        case 111: RINV(613); break;
        case 112: RINV(617); break;
        case 113: RINV(619); break;
        case 114: RINV(631); break;
        case 115: RINV(641); break;
        case 116: RINV(643); break;
        case 117: RINV(647); break;
        case 118: RINV(653); break;
        case 119: RINV(659); break;
        case 120: RINV(661); break;
        case 121: RINV(673); break;
        case 122: RINV(677); break;
        case 123: RINV(683); break;
        case 124: RINV(691); break;
        case 125: RINV(701); break;
        case 126: RINV(709); break;
        case 127: RINV(719); break;
        case 128: RINV(727); break;
        case 129: RINV(733); break;
        case 130: RINV(739); break;
        case 131: RINV(743); break;
        case 132: RINV(751); break;
        case 133: RINV(757); break;
        case 134: RINV(761); break;
        case 135: RINV(769); break;
        case 136: RINV(773); break;
        case 137: RINV(787); break;
        case 138: RINV(797); break;
        case 139: RINV(809); break;
        case 140: RINV(811); break;
        case 141: RINV(821); break;
        case 142: RINV(823); break;
        case 143: RINV(827); break;
        case 144: RINV(829); break;
        case 145: RINV(839); break;
        case 146: RINV(853); break;
        case 147: RINV(857); break;
        case 148: RINV(859); break;
        case 149: RINV(863); break;
        case 150: RINV(877); break;
        case 151: RINV(881); break;
        case 152: RINV(883); break;
        case 153: RINV(887); break;
        case 154: RINV(907); break;
        case 155: RINV(911); break;
        case 156: RINV(919); break;
        case 157: RINV(929); break;
        case 158: RINV(937); break;
        case 159: RINV(941); break;
        case 160: RINV(947); break;
        case 161: RINV(953); break;
        case 162: RINV(967); break;
        case 163: RINV(971); break;
        case 164: RINV(977); break;
        case 165: RINV(983); break;
        case 166: RINV(991); break;
        case 167: RINV(997); break;
        case 168: RINV(1009); break;
        case 169: RINV(1013); break;
        case 170: RINV(1019); break;
        case 171: RINV(1021); break;
        case 172: RINV(1031); break;
        case 173: RINV(1033); break;
        case 174: RINV(1039); break;
        case 175: RINV(1049); break;
        case 176: RINV(1051); break;
        case 177: RINV(1061); break;
        case 178: RINV(1063); break;
        case 179: RINV(1069); break;
        case 180: RINV(1087); break;
        case 181: RINV(1091); break;
        case 182: RINV(1093); break;
        case 183: RINV(1097); break;
        case 184: RINV(1103); break;
        case 185: RINV(1109); break;
        case 186: RINV(1117); break;
        case 187: RINV(1123); break;
        case 188: RINV(1129); break;
        case 189: RINV(1151); break;
        case 190: RINV(1153); break;
        case 191: RINV(1163); break;
        case 192: RINV(1171); break;
        case 193: RINV(1181); break;
        case 194: RINV(1187); break;
        case 195: RINV(1193); break;
        case 196: RINV(1201); break;
        case 197: RINV(1213); break;
        case 198: RINV(1217); break;
        case 199: RINV(1223); break;
        case 200: RINV(1229); break;
        case 201: RINV(1231); break;
        case 202: RINV(1237); break;
        case 203: RINV(1249); break;
        case 204: RINV(1259); break;
        case 205: RINV(1277); break;
        case 206: RINV(1279); break;
        case 207: RINV(1283); break;
        case 208: RINV(1289); break;
        case 209: RINV(1291); break;
        case 210: RINV(1297); break;
        case 211: RINV(1301); break;
        case 212: RINV(1303); break;
        case 213: RINV(1307); break;
        case 214: RINV(1319); break;
        case 215: RINV(1321); break;
        case 216: RINV(1327); break;
        case 217: RINV(1361); break;
        case 218: RINV(1367); break;
        case 219: RINV(1373); break;
        case 220: RINV(1381); break;
        case 221: RINV(1399); break;
        case 222: RINV(1409); break;
        case 223: RINV(1423); break;
        case 224: RINV(1427); break;
        case 225: RINV(1429); break;
        case 226: RINV(1433); break;
        case 227: RINV(1439); break;
        case 228: RINV(1447); break;
        case 229: RINV(1451); break;
        case 230: RINV(1453); break;
        case 231: RINV(1459); break;
        case 232: RINV(1471); break;
        case 233: RINV(1481); break;
        case 234: RINV(1483); break;
        case 235: RINV(1487); break;
        case 236: RINV(1489); break;
        case 237: RINV(1493); break;
        case 238: RINV(1499); break;
        case 239: RINV(1511); break;
        case 240: RINV(1523); break;
        case 241: RINV(1531); break;
        case 242: RINV(1543); break;
        case 243: RINV(1549); break;
        case 244: RINV(1553); break;
        case 245: RINV(1559); break;
        case 246: RINV(1567); break;
        case 247: RINV(1571); break;
        case 248: RINV(1579); break;
        case 249: RINV(1583); break;
        case 250: RINV(1597); break;
        case 251: RINV(1601); break;
        case 252: RINV(1607); break;
        case 253: RINV(1609); break;
        case 254: RINV(1613); break;
        case 255: RINV(1619); break;
        case 256: RINV(1621); break;
        case 257: RINV(1627); break;
        case 258: RINV(1637); break;
        case 259: RINV(1657); break;
        case 260: RINV(1663); break;
        case 261: RINV(1667); break;
        case 262: RINV(1669); break;
        case 263: RINV(1693); break;
        case 264: RINV(1697); break;
        case 265: RINV(1699); break;
        case 266: RINV(1709); break;
        case 267: RINV(1721); break;
        case 268: RINV(1723); break;
        case 269: RINV(1733); break;
        case 270: RINV(1741); break;
        case 271: RINV(1747); break;
        case 272: RINV(1753); break;
        case 273: RINV(1759); break;
        case 274: RINV(1777); break;
        case 275: RINV(1783); break;
        case 276: RINV(1787); break;
        case 277: RINV(1789); break;
        case 278: RINV(1801); break;
        case 279: RINV(1811); break;
        case 280: RINV(1823); break;
        case 281: RINV(1831); break;
        case 282: RINV(1847); break;
        case 283: RINV(1861); break;
        case 284: RINV(1867); break;
        case 285: RINV(1871); break;
        case 286: RINV(1873); break;
        case 287: RINV(1877); break;
        case 288: RINV(1879); break;
        case 289: RINV(1889); break;
        case 290: RINV(1901); break;
        case 291: RINV(1907); break;
        case 292: RINV(1913); break;
        case 293: RINV(1931); break;
        case 294: RINV(1933); break;
        case 295: RINV(1949); break;
        case 296: RINV(1951); break;
        case 297: RINV(1973); break;
        case 298: RINV(1979); break;
        case 299: RINV(1987); break;
        case 300: RINV(1993); break;
        case 301: RINV(1997); break;
        case 302: RINV(1999); break;
        case 303: RINV(2003); break;
        case 304: RINV(2011); break;
        case 305: RINV(2017); break;
        case 306: RINV(2027); break;
        case 307: RINV(2029); break;
        case 308: RINV(2039); break;
        case 309: RINV(2053); break;
        case 310: RINV(2063); break;
        case 311: RINV(2069); break;
        case 312: RINV(2081); break;
        case 313: RINV(2083); break;
        case 314: RINV(2087); break;
        case 315: RINV(2089); break;
        case 316: RINV(2099); break;
        case 317: RINV(2111); break;
        case 318: RINV(2113); break;
        case 319: RINV(2129); break;
        case 320: RINV(2131); break;
        case 321: RINV(2137); break;
        case 322: RINV(2141); break;
        case 323: RINV(2143); break;
        case 324: RINV(2153); break;
        case 325: RINV(2161); break;
        case 326: RINV(2179); break;
        case 327: RINV(2203); break;
        case 328: RINV(2207); break;
        case 329: RINV(2213); break;
        case 330: RINV(2221); break;
        case 331: RINV(2237); break;
        case 332: RINV(2239); break;
        case 333: RINV(2243); break;
        case 334: RINV(2251); break;
        case 335: RINV(2267); break;
        case 336: RINV(2269); break;
        case 337: RINV(2273); break;
        case 338: RINV(2281); break;
        case 339: RINV(2287); break;
        case 340: RINV(2293); break;
        case 341: RINV(2297); break;
        case 342: RINV(2309); break;
        case 343: RINV(2311); break;
        case 344: RINV(2333); break;
        case 345: RINV(2339); break;
        case 346: RINV(2341); break;
        case 347: RINV(2347); break;
        case 348: RINV(2351); break;
        case 349: RINV(2357); break;
        case 350: RINV(2371); break;
        case 351: RINV(2377); break;
        case 352: RINV(2381); break;
        case 353: RINV(2383); break;
        case 354: RINV(2389); break;
        case 355: RINV(2393); break;
        case 356: RINV(2399); break;
        case 357: RINV(2411); break;
        case 358: RINV(2417); break;
        case 359: RINV(2423); break;
        case 360: RINV(2437); break;
        case 361: RINV(2441); break;
        case 362: RINV(2447); break;
        case 363: RINV(2459); break;
        case 364: RINV(2467); break;
        case 365: RINV(2473); break;
        case 366: RINV(2477); break;
        case 367: RINV(2503); break;
        case 368: RINV(2521); break;
        case 369: RINV(2531); break;
        case 370: RINV(2539); break;
        case 371: RINV(2543); break;
        case 372: RINV(2549); break;
        case 373: RINV(2551); break;
        case 374: RINV(2557); break;
        case 375: RINV(2579); break;
        case 376: RINV(2591); break;
        case 377: RINV(2593); break;
        case 378: RINV(2609); break;
        case 379: RINV(2617); break;
        case 380: RINV(2621); break;
        case 381: RINV(2633); break;
        case 382: RINV(2647); break;
        case 383: RINV(2657); break;
        case 384: RINV(2659); break;
        case 385: RINV(2663); break;
        case 386: RINV(2671); break;
        case 387: RINV(2677); break;
        case 388: RINV(2683); break;
        case 389: RINV(2687); break;
        case 390: RINV(2689); break;
        case 391: RINV(2693); break;
        case 392: RINV(2699); break;
        case 393: RINV(2707); break;
        case 394: RINV(2711); break;
        case 395: RINV(2713); break;
        case 396: RINV(2719); break;
        case 397: RINV(2729); break;
        case 398: RINV(2731); break;
        case 399: RINV(2741); break;
        case 400: RINV(2749); break;
        case 401: RINV(2753); break;
        case 402: RINV(2767); break;
        case 403: RINV(2777); break;
        case 404: RINV(2789); break;
        case 405: RINV(2791); break;
        case 406: RINV(2797); break;
        case 407: RINV(2801); break;
        case 408: RINV(2803); break;
        case 409: RINV(2819); break;
        case 410: RINV(2833); break;
        case 411: RINV(2837); break;
        case 412: RINV(2843); break;
        case 413: RINV(2851); break;
        case 414: RINV(2857); break;
        case 415: RINV(2861); break;
        case 416: RINV(2879); break;
        case 417: RINV(2887); break;
        case 418: RINV(2897); break;
        case 419: RINV(2903); break;
        case 420: RINV(2909); break;
        case 421: RINV(2917); break;
        case 422: RINV(2927); break;
        case 423: RINV(2939); break;
        case 424: RINV(2953); break;
        case 425: RINV(2957); break;
        case 426: RINV(2963); break;
        case 427: RINV(2969); break;
        case 428: RINV(2971); break;
        case 429: RINV(2999); break;
        case 430: RINV(3001); break;
        case 431: RINV(3011); break;
        case 432: RINV(3019); break;
        case 433: RINV(3023); break;
        case 434: RINV(3037); break;
        case 435: RINV(3041); break;
        case 436: RINV(3049); break;
        case 437: RINV(3061); break;
        case 438: RINV(3067); break;
        case 439: RINV(3079); break;
        case 440: RINV(3083); break;
        case 441: RINV(3089); break;
        case 442: RINV(3109); break;
        case 443: RINV(3119); break;
        case 444: RINV(3121); break;
        case 445: RINV(3137); break;
        case 446: RINV(3163); break;
        case 447: RINV(3167); break;
        case 448: RINV(3169); break;
        case 449: RINV(3181); break;
        case 450: RINV(3187); break;
        case 451: RINV(3191); break;
        case 452: RINV(3203); break;
        case 453: RINV(3209); break;
        case 454: RINV(3217); break;
        case 455: RINV(3221); break;
        case 456: RINV(3229); break;
        case 457: RINV(3251); break;
        case 458: RINV(3253); break;
        case 459: RINV(3257); break;
        case 460: RINV(3259); break;
        case 461: RINV(3271); break;
        case 462: RINV(3299); break;
        case 463: RINV(3301); break;
        case 464: RINV(3307); break;
        case 465: RINV(3313); break;
        case 466: RINV(3319); break;
        case 467: RINV(3323); break;
        case 468: RINV(3329); break;
        case 469: RINV(3331); break;
        case 470: RINV(3343); break;
        case 471: RINV(3347); break;
        case 472: RINV(3359); break;
        case 473: RINV(3361); break;
        case 474: RINV(3371); break;
        case 475: RINV(3373); break;
        case 476: RINV(3389); break;
        case 477: RINV(3391); break;
        case 478: RINV(3407); break;
        case 479: RINV(3413); break;
        case 480: RINV(3433); break;
        case 481: RINV(3449); break;
        case 482: RINV(3457); break;
        case 483: RINV(3461); break;
        case 484: RINV(3463); break;
        case 485: RINV(3467); break;
        case 486: RINV(3469); break;
        case 487: RINV(3491); break;
        case 488: RINV(3499); break;
        case 489: RINV(3511); break;
        case 490: RINV(3517); break;
        case 491: RINV(3527); break;
        case 492: RINV(3529); break;
        case 493: RINV(3533); break;
        case 494: RINV(3539); break;
        case 495: RINV(3541); break;
        case 496: RINV(3547); break;
        case 497: RINV(3557); break;
        case 498: RINV(3559); break;
        case 499: RINV(3571); break;
        case 500: RINV(3581); break;
        case 501: RINV(3583); break;
        case 502: RINV(3593); break;
        case 503: RINV(3607); break;
        case 504: RINV(3613); break;
        case 505: RINV(3617); break;
        case 506: RINV(3623); break;
        case 507: RINV(3631); break;
        case 508: RINV(3637); break;
        case 509: RINV(3643); break;
        case 510: RINV(3659); break;
        case 511: RINV(3671); break;
        case 512: RINV(3673); break;
        case 513: RINV(3677); break;
        case 514: RINV(3691); break;
        case 515: RINV(3697); break;
        case 516: RINV(3701); break;
        case 517: RINV(3709); break;
        case 518: RINV(3719); break;
        case 519: RINV(3727); break;
        case 520: RINV(3733); break;
        case 521: RINV(3739); break;
        case 522: RINV(3761); break;
        case 523: RINV(3767); break;
        case 524: RINV(3769); break;
        case 525: RINV(3779); break;
        case 526: RINV(3793); break;
        case 527: RINV(3797); break;
        case 528: RINV(3803); break;
        case 529: RINV(3821); break;
        case 530: RINV(3823); break;
        case 531: RINV(3833); break;
        case 532: RINV(3847); break;
        case 533: RINV(3851); break;
        case 534: RINV(3853); break;
        case 535: RINV(3863); break;
        case 536: RINV(3877); break;
        case 537: RINV(3881); break;
        case 538: RINV(3889); break;
        case 539: RINV(3907); break;
        case 540: RINV(3911); break;
        case 541: RINV(3917); break;
        case 542: RINV(3919); break;
        case 543: RINV(3923); break;
        case 544: RINV(3929); break;
        case 545: RINV(3931); break;
        case 546: RINV(3943); break;
        case 547: RINV(3947); break;
        case 548: RINV(3967); break;
        case 549: RINV(3989); break;
        case 550: RINV(4001); break;
        case 551: RINV(4003); break;
        case 552: RINV(4007); break;
        case 553: RINV(4013); break;
        case 554: RINV(4019); break;
        case 555: RINV(4021); break;
        case 556: RINV(4027); break;
        case 557: RINV(4049); break;
        case 558: RINV(4051); break;
        case 559: RINV(4057); break;
        case 560: RINV(4073); break;
        case 561: RINV(4079); break;
        case 562: RINV(4091); break;
        case 563: RINV(4093); break;
        case 564: RINV(4099); break;
        case 565: RINV(4111); break;
        case 566: RINV(4127); break;
        case 567: RINV(4129); break;
        case 568: RINV(4133); break;
        case 569: RINV(4139); break;
        case 570: RINV(4153); break;
        case 571: RINV(4157); break;
        case 572: RINV(4159); break;
        case 573: RINV(4177); break;
        case 574: RINV(4201); break;
        case 575: RINV(4211); break;
        case 576: RINV(4217); break;
        case 577: RINV(4219); break;
        case 578: RINV(4229); break;
        case 579: RINV(4231); break;
        case 580: RINV(4241); break;
        case 581: RINV(4243); break;
        case 582: RINV(4253); break;
        case 583: RINV(4259); break;
        case 584: RINV(4261); break;
        case 585: RINV(4271); break;
        case 586: RINV(4273); break;
        case 587: RINV(4283); break;
        case 588: RINV(4289); break;
        case 589: RINV(4297); break;
        case 590: RINV(4327); break;
        case 591: RINV(4337); break;
        case 592: RINV(4339); break;
        case 593: RINV(4349); break;
        case 594: RINV(4357); break;
        case 595: RINV(4363); break;
        case 596: RINV(4373); break;
        case 597: RINV(4391); break;
        case 598: RINV(4397); break;
        case 599: RINV(4409); break;
        case 600: RINV(4421); break;
        case 601: RINV(4423); break;
        case 602: RINV(4441); break;
        case 603: RINV(4447); break;
        case 604: RINV(4451); break;
        case 605: RINV(4457); break;
        case 606: RINV(4463); break;
        case 607: RINV(4481); break;
        case 608: RINV(4483); break;
        case 609: RINV(4493); break;
        case 610: RINV(4507); break;
        case 611: RINV(4513); break;
        case 612: RINV(4517); break;
        case 613: RINV(4519); break;
        case 614: RINV(4523); break;
        case 615: RINV(4547); break;
        case 616: RINV(4549); break;
        case 617: RINV(4561); break;
        case 618: RINV(4567); break;
        case 619: RINV(4583); break;
        case 620: RINV(4591); break;
        case 621: RINV(4597); break;
        case 622: RINV(4603); break;
        case 623: RINV(4621); break;
        case 624: RINV(4637); break;
        case 625: RINV(4639); break;
        case 626: RINV(4643); break;
        case 627: RINV(4649); break;
        case 628: RINV(4651); break;
        case 629: RINV(4657); break;
        case 630: RINV(4663); break;
        case 631: RINV(4673); break;
        case 632: RINV(4679); break;
        case 633: RINV(4691); break;
        case 634: RINV(4703); break;
        case 635: RINV(4721); break;
        case 636: RINV(4723); break;
        case 637: RINV(4729); break;
        case 638: RINV(4733); break;
        case 639: RINV(4751); break;
        case 640: RINV(4759); break;
        case 641: RINV(4783); break;
        case 642: RINV(4787); break;
        case 643: RINV(4789); break;
        case 644: RINV(4793); break;
        case 645: RINV(4799); break;
        case 646: RINV(4801); break;
        case 647: RINV(4813); break;
        case 648: RINV(4817); break;
        case 649: RINV(4831); break;
        case 650: RINV(4861); break;
        case 651: RINV(4871); break;
        case 652: RINV(4877); break;
        case 653: RINV(4889); break;
        case 654: RINV(4903); break;
        case 655: RINV(4909); break;
        case 656: RINV(4919); break;
        case 657: RINV(4931); break;
        case 658: RINV(4933); break;
        case 659: RINV(4937); break;
        case 660: RINV(4943); break;
        case 661: RINV(4951); break;
        case 662: RINV(4957); break;
        case 663: RINV(4967); break;
        case 664: RINV(4969); break;
        case 665: RINV(4973); break;
        case 666: RINV(4987); break;
        case 667: RINV(4993); break;
        case 668: RINV(4999); break;
        case 669: RINV(5003); break;
        case 670: RINV(5009); break;
        case 671: RINV(5011); break;
        case 672: RINV(5021); break;
        case 673: RINV(5023); break;
        case 674: RINV(5039); break;
        case 675: RINV(5051); break;
        case 676: RINV(5059); break;
        case 677: RINV(5077); break;
        case 678: RINV(5081); break;
        case 679: RINV(5087); break;
        case 680: RINV(5099); break;
        case 681: RINV(5101); break;
        case 682: RINV(5107); break;
        case 683: RINV(5113); break;
        case 684: RINV(5119); break;
        case 685: RINV(5147); break;
        case 686: RINV(5153); break;
        case 687: RINV(5167); break;
        case 688: RINV(5171); break;
        case 689: RINV(5179); break;
        case 690: RINV(5189); break;
        case 691: RINV(5197); break;
        case 692: RINV(5209); break;
        case 693: RINV(5227); break;
        case 694: RINV(5231); break;
        case 695: RINV(5233); break;
        case 696: RINV(5237); break;
        case 697: RINV(5261); break;
        case 698: RINV(5273); break;
        case 699: RINV(5279); break;
        case 700: RINV(5281); break;
        case 701: RINV(5297); break;
        case 702: RINV(5303); break;
        case 703: RINV(5309); break;
        case 704: RINV(5323); break;
        case 705: RINV(5333); break;
        case 706: RINV(5347); break;
        case 707: RINV(5351); break;
        case 708: RINV(5381); break;
        case 709: RINV(5387); break;
        case 710: RINV(5393); break;
        case 711: RINV(5399); break;
        case 712: RINV(5407); break;
        case 713: RINV(5413); break;
        case 714: RINV(5417); break;
        case 715: RINV(5419); break;
        case 716: RINV(5431); break;
        case 717: RINV(5437); break;
        case 718: RINV(5441); break;
        case 719: RINV(5443); break;
        case 720: RINV(5449); break;
        case 721: RINV(5471); break;
        case 722: RINV(5477); break;
        case 723: RINV(5479); break;
        case 724: RINV(5483); break;
        case 725: RINV(5501); break;
        case 726: RINV(5503); break;
        case 727: RINV(5507); break;
        case 728: RINV(5519); break;
        case 729: RINV(5521); break;
        case 730: RINV(5527); break;
        case 731: RINV(5531); break;
        case 732: RINV(5557); break;
        case 733: RINV(5563); break;
        case 734: RINV(5569); break;
        case 735: RINV(5573); break;
        case 736: RINV(5581); break;
        case 737: RINV(5591); break;
        case 738: RINV(5623); break;
        case 739: RINV(5639); break;
        case 740: RINV(5641); break;
        case 741: RINV(5647); break;
        case 742: RINV(5651); break;
        case 743: RINV(5653); break;
        case 744: RINV(5657); break;
        case 745: RINV(5659); break;
        case 746: RINV(5669); break;
        case 747: RINV(5683); break;
        case 748: RINV(5689); break;
        case 749: RINV(5693); break;
        case 750: RINV(5701); break;
        case 751: RINV(5711); break;
        case 752: RINV(5717); break;
        case 753: RINV(5737); break;
        case 754: RINV(5741); break;
        case 755: RINV(5743); break;
        case 756: RINV(5749); break;
        case 757: RINV(5779); break;
        case 758: RINV(5783); break;
        case 759: RINV(5791); break;
        case 760: RINV(5801); break;
        case 761: RINV(5807); break;
        case 762: RINV(5813); break;
        case 763: RINV(5821); break;
        case 764: RINV(5827); break;
        case 765: RINV(5839); break;
        case 766: RINV(5843); break;
        case 767: RINV(5849); break;
        case 768: RINV(5851); break;
        case 769: RINV(5857); break;
        case 770: RINV(5861); break;
        case 771: RINV(5867); break;
        case 772: RINV(5869); break;
        case 773: RINV(5879); break;
        case 774: RINV(5881); break;
        case 775: RINV(5897); break;
        case 776: RINV(5903); break;
        case 777: RINV(5923); break;
        case 778: RINV(5927); break;
        case 779: RINV(5939); break;
        case 780: RINV(5953); break;
        case 781: RINV(5981); break;
        case 782: RINV(5987); break;
        case 783: RINV(6007); break;
        case 784: RINV(6011); break;
        case 785: RINV(6029); break;
        case 786: RINV(6037); break;
        case 787: RINV(6043); break;
        case 788: RINV(6047); break;
        case 789: RINV(6053); break;
        case 790: RINV(6067); break;
        case 791: RINV(6073); break;
        case 792: RINV(6079); break;
        case 793: RINV(6089); break;
        case 794: RINV(6091); break;
        case 795: RINV(6101); break;
        case 796: RINV(6113); break;
        case 797: RINV(6121); break;
        case 798: RINV(6131); break;
        case 799: RINV(6133); break;
        case 800: RINV(6143); break;
        case 801: RINV(6151); break;
        case 802: RINV(6163); break;
        case 803: RINV(6173); break;
        case 804: RINV(6197); break;
        case 805: RINV(6199); break;
        case 806: RINV(6203); break;
        case 807: RINV(6211); break;
        case 808: RINV(6217); break;
        case 809: RINV(6221); break;
        case 810: RINV(6229); break;
        case 811: RINV(6247); break;
        case 812: RINV(6257); break;
        case 813: RINV(6263); break;
        case 814: RINV(6269); break;
        case 815: RINV(6271); break;
        case 816: RINV(6277); break;
        case 817: RINV(6287); break;
        case 818: RINV(6299); break;
        case 819: RINV(6301); break;
        case 820: RINV(6311); break;
        case 821: RINV(6317); break;
        case 822: RINV(6323); break;
        case 823: RINV(6329); break;
        case 824: RINV(6337); break;
        case 825: RINV(6343); break;
        case 826: RINV(6353); break;
        case 827: RINV(6359); break;
        case 828: RINV(6361); break;
        case 829: RINV(6367); break;
        case 830: RINV(6373); break;
        case 831: RINV(6379); break;
        case 832: RINV(6389); break;
        case 833: RINV(6397); break;
        case 834: RINV(6421); break;
        case 835: RINV(6427); break;
        case 836: RINV(6449); break;
        case 837: RINV(6451); break;
        case 838: RINV(6469); break;
        case 839: RINV(6473); break;
        case 840: RINV(6481); break;
        case 841: RINV(6491); break;
        case 842: RINV(6521); break;
        case 843: RINV(6529); break;
        case 844: RINV(6547); break;
        case 845: RINV(6551); break;
        case 846: RINV(6553); break;
        case 847: RINV(6563); break;
        case 848: RINV(6569); break;
        case 849: RINV(6571); break;
        case 850: RINV(6577); break;
        case 851: RINV(6581); break;
        case 852: RINV(6599); break;
        case 853: RINV(6607); break;
        case 854: RINV(6619); break;
        case 855: RINV(6637); break;
        case 856: RINV(6653); break;
        case 857: RINV(6659); break;
        case 858: RINV(6661); break;
        case 859: RINV(6673); break;
        case 860: RINV(6679); break;
        case 861: RINV(6689); break;
        case 862: RINV(6691); break;
        case 863: RINV(6701); break;
        case 864: RINV(6703); break;
        case 865: RINV(6709); break;
        case 866: RINV(6719); break;
        case 867: RINV(6733); break;
        case 868: RINV(6737); break;
        case 869: RINV(6761); break;
        case 870: RINV(6763); break;
        case 871: RINV(6779); break;
        case 872: RINV(6781); break;
        case 873: RINV(6791); break;
        case 874: RINV(6793); break;
        case 875: RINV(6803); break;
        case 876: RINV(6823); break;
        case 877: RINV(6827); break;
        case 878: RINV(6829); break;
        case 879: RINV(6833); break;
        case 880: RINV(6841); break;
        case 881: RINV(6857); break;
        case 882: RINV(6863); break;
        case 883: RINV(6869); break;
        case 884: RINV(6871); break;
        case 885: RINV(6883); break;
        case 886: RINV(6899); break;
        case 887: RINV(6907); break;
        case 888: RINV(6911); break;
        case 889: RINV(6917); break;
        case 890: RINV(6947); break;
        case 891: RINV(6949); break;
        case 892: RINV(6959); break;
        case 893: RINV(6961); break;
        case 894: RINV(6967); break;
        case 895: RINV(6971); break;
        case 896: RINV(6977); break;
        case 897: RINV(6983); break;
        case 898: RINV(6991); break;
        case 899: RINV(6997); break;
        case 900: RINV(7001); break;
        case 901: RINV(7013); break;
        case 902: RINV(7019); break;
        case 903: RINV(7027); break;
        case 904: RINV(7039); break;
        case 905: RINV(7043); break;
        case 906: RINV(7057); break;
        case 907: RINV(7069); break;
        case 908: RINV(7079); break;
        case 909: RINV(7103); break;
        case 910: RINV(7109); break;
        case 911: RINV(7121); break;
        case 912: RINV(7127); break;
        case 913: RINV(7129); break;
        case 914: RINV(7151); break;
        case 915: RINV(7159); break;
        case 916: RINV(7177); break;
        case 917: RINV(7187); break;
        case 918: RINV(7193); break;
        case 919: RINV(7207); break;
        case 920: RINV(7211); break;
        case 921: RINV(7213); break;
        case 922: RINV(7219); break;
        case 923: RINV(7229); break;
        case 924: RINV(7237); break;
        case 925: RINV(7243); break;
        case 926: RINV(7247); break;
        case 927: RINV(7253); break;
        case 928: RINV(7283); break;
        case 929: RINV(7297); break;
        case 930: RINV(7307); break;
        case 931: RINV(7309); break;
        case 932: RINV(7321); break;
        case 933: RINV(7331); break;
        case 934: RINV(7333); break;
        case 935: RINV(7349); break;
        case 936: RINV(7351); break;
        case 937: RINV(7369); break;
        case 938: RINV(7393); break;
        case 939: RINV(7411); break;
        case 940: RINV(7417); break;
        case 941: RINV(7433); break;
        case 942: RINV(7451); break;
        case 943: RINV(7457); break;
        case 944: RINV(7459); break;
        case 945: RINV(7477); break;
        case 946: RINV(7481); break;
        case 947: RINV(7487); break;
        case 948: RINV(7489); break;
        case 949: RINV(7499); break;
        case 950: RINV(7507); break;
        case 951: RINV(7517); break;
        case 952: RINV(7523); break;
        case 953: RINV(7529); break;
        case 954: RINV(7537); break;
        case 955: RINV(7541); break;
        case 956: RINV(7547); break;
        case 957: RINV(7549); break;
        case 958: RINV(7559); break;
        case 959: RINV(7561); break;
        case 960: RINV(7573); break;
        case 961: RINV(7577); break;
        case 962: RINV(7583); break;
        case 963: RINV(7589); break;
        case 964: RINV(7591); break;
        case 965: RINV(7603); break;
        case 966: RINV(7607); break;
        case 967: RINV(7621); break;
        case 968: RINV(7639); break;
        case 969: RINV(7643); break;
        case 970: RINV(7649); break;
        case 971: RINV(7669); break;
        case 972: RINV(7673); break;
        case 973: RINV(7681); break;
        case 974: RINV(7687); break;
        case 975: RINV(7691); break;
        case 976: RINV(7699); break;
        case 977: RINV(7703); break;
        case 978: RINV(7717); break;
        case 979: RINV(7723); break;
        case 980: RINV(7727); break;
        case 981: RINV(7741); break;
        case 982: RINV(7753); break;
        case 983: RINV(7757); break;
        case 984: RINV(7759); break;
        case 985: RINV(7789); break;
        case 986: RINV(7793); break;
        case 987: RINV(7817); break;
        case 988: RINV(7823); break;
        case 989: RINV(7829); break;
        case 990: RINV(7841); break;
        case 991: RINV(7853); break;
        case 992: RINV(7867); break;
        case 993: RINV(7873); break;
        case 994: RINV(7877); break;
        case 995: RINV(7879); break;
        case 996: RINV(7883); break;
        case 997: RINV(7901); break;
        case 998: RINV(7907); break;
        case 999: RINV(7919); break;
        case 1000: RINV(7927); break;
        case 1001: RINV(7933); break;
        case 1002: RINV(7937); break;
        case 1003: RINV(7949); break;
        case 1004: RINV(7951); break;
        case 1005: RINV(7963); break;
        case 1006: RINV(7993); break;
        case 1007: RINV(8009); break;
        case 1008: RINV(8011); break;
        case 1009: RINV(8017); break;
        case 1010: RINV(8039); break;
        case 1011: RINV(8053); break;
        case 1012: RINV(8059); break;
        case 1013: RINV(8069); break;
        case 1014: RINV(8081); break;
        case 1015: RINV(8087); break;
        case 1016: RINV(8089); break;
        case 1017: RINV(8093); break;
        case 1018: RINV(8101); break;
        case 1019: RINV(8111); break;
        case 1020: RINV(8117); break;
        case 1021: RINV(8123); break;
        case 1022: RINV(8147); break;
        case 1023: RINV(8161); break;
    }
    return std::min(inverse, ONE_MINUS_EPS);
}

Float scrambledRadicalInverseFast(uint16_t baseIndex, uint64_t index, uint16_t *perm) {
    Float inverse = 0.0f;

    switch (baseIndex) {
        case 0: SCRAMBLED_RINV(2); break;
        case 1: SCRAMBLED_RINV(3); break;
        case 2: SCRAMBLED_RINV(5); break;
        case 3: SCRAMBLED_RINV(7); break;
        case 4: SCRAMBLED_RINV(11); break;
        case 5: SCRAMBLED_RINV(13); break;
        case 6: SCRAMBLED_RINV(17); break;
        case 7: SCRAMBLED_RINV(19); break;
        case 8: SCRAMBLED_RINV(23); break;
        case 9: SCRAMBLED_RINV(29); break;
        case 10: SCRAMBLED_RINV(31); break;
        case 11: SCRAMBLED_RINV(37); break;
        case 12: SCRAMBLED_RINV(41); break;
        case 13: SCRAMBLED_RINV(43); break;
        case 14: SCRAMBLED_RINV(47); break;
        case 15: SCRAMBLED_RINV(53); break;
        case 16: SCRAMBLED_RINV(59); break;
        case 17: SCRAMBLED_RINV(61); break;
        case 18: SCRAMBLED_RINV(67); break;
        case 19: SCRAMBLED_RINV(71); break;
        case 20: SCRAMBLED_RINV(73); break;
        case 21: SCRAMBLED_RINV(79); break;
        case 22: SCRAMBLED_RINV(83); break;
        case 23: SCRAMBLED_RINV(89); break;
        case 24: SCRAMBLED_RINV(97); break;
        case 25: SCRAMBLED_RINV(101); break;
        case 26: SCRAMBLED_RINV(103); break;
        case 27: SCRAMBLED_RINV(107); break;
        case 28: SCRAMBLED_RINV(109); break;
        case 29: SCRAMBLED_RINV(113); break;
        case 30: SCRAMBLED_RINV(127); break;
        case 31: SCRAMBLED_RINV(131); break;
        case 32: SCRAMBLED_RINV(137); break;
        case 33: SCRAMBLED_RINV(139); break;
        case 34: SCRAMBLED_RINV(149); break;
        case 35: SCRAMBLED_RINV(151); break;
        case 36: SCRAMBLED_RINV(157); break;
        case 37: SCRAMBLED_RINV(163); break;
        case 38: SCRAMBLED_RINV(167); break;
        case 39: SCRAMBLED_RINV(173); break;
        case 40: SCRAMBLED_RINV(179); break;
        case 41: SCRAMBLED_RINV(181); break;
        case 42: SCRAMBLED_RINV(191); break;
        case 43: SCRAMBLED_RINV(193); break;
        case 44: SCRAMBLED_RINV(197); break;
        case 45: SCRAMBLED_RINV(199); break;
        case 46: SCRAMBLED_RINV(211); break;
        case 47: SCRAMBLED_RINV(223); break;
        case 48: SCRAMBLED_RINV(227); break;
        case 49: SCRAMBLED_RINV(229); break;
        case 50: SCRAMBLED_RINV(233); break;
        case 51: SCRAMBLED_RINV(239); break;
        case 52: SCRAMBLED_RINV(241); break;
        case 53: SCRAMBLED_RINV(251); break;
        case 54: SCRAMBLED_RINV(257); break;
        case 55: SCRAMBLED_RINV(263); break;
        case 56: SCRAMBLED_RINV(269); break;
        case 57: SCRAMBLED_RINV(271); break;
        case 58: SCRAMBLED_RINV(277); break;
        case 59: SCRAMBLED_RINV(281); break;
        case 60: SCRAMBLED_RINV(283); break;
        case 61: SCRAMBLED_RINV(293); break;
        case 62: SCRAMBLED_RINV(307); break;
        case 63: SCRAMBLED_RINV(311); break;
        case 64: SCRAMBLED_RINV(313); break;
        case 65: SCRAMBLED_RINV(317); break;
        case 66: SCRAMBLED_RINV(331); break;
        case 67: SCRAMBLED_RINV(337); break;
        case 68: SCRAMBLED_RINV(347); break;
        case 69: SCRAMBLED_RINV(349); break;
        case 70: SCRAMBLED_RINV(353); break;
        case 71: SCRAMBLED_RINV(359); break;
        case 72: SCRAMBLED_RINV(367); break;
        case 73: SCRAMBLED_RINV(373); break;
        case 74: SCRAMBLED_RINV(379); break;
        case 75: SCRAMBLED_RINV(383); break;
        case 76: SCRAMBLED_RINV(389); break;
        case 77: SCRAMBLED_RINV(397); break;
        case 78: SCRAMBLED_RINV(401); break;
        case 79: SCRAMBLED_RINV(409); break;
        case 80: SCRAMBLED_RINV(419); break;
        case 81: SCRAMBLED_RINV(421); break;
        case 82: SCRAMBLED_RINV(431); break;
        case 83: SCRAMBLED_RINV(433); break;
        case 84: SCRAMBLED_RINV(439); break;
        case 85: SCRAMBLED_RINV(443); break;
        case 86: SCRAMBLED_RINV(449); break;
        case 87: SCRAMBLED_RINV(457); break;
        case 88: SCRAMBLED_RINV(461); break;
        case 89: SCRAMBLED_RINV(463); break;
        case 90: SCRAMBLED_RINV(467); break;
        case 91: SCRAMBLED_RINV(479); break;
        case 92: SCRAMBLED_RINV(487); break;
        case 93: SCRAMBLED_RINV(491); break;
        case 94: SCRAMBLED_RINV(499); break;
        case 95: SCRAMBLED_RINV(503); break;
        case 96: SCRAMBLED_RINV(509); break;
        case 97: SCRAMBLED_RINV(521); break;
        case 98: SCRAMBLED_RINV(523); break;
        case 99: SCRAMBLED_RINV(541); break;
        case 100: SCRAMBLED_RINV(547); break;
        case 101: SCRAMBLED_RINV(557); break;
        case 102: SCRAMBLED_RINV(563); break;
        case 103: SCRAMBLED_RINV(569); break;
        case 104: SCRAMBLED_RINV(571); break;
        case 105: SCRAMBLED_RINV(577); break;
        case 106: SCRAMBLED_RINV(587); break;
        case 107: SCRAMBLED_RINV(593); break;
        case 108: SCRAMBLED_RINV(599); break;
        case 109: SCRAMBLED_RINV(601); break;
        case 110: SCRAMBLED_RINV(607); break;
        case 111: SCRAMBLED_RINV(613); break;
        case 112: SCRAMBLED_RINV(617); break;
        case 113: SCRAMBLED_RINV(619); break;
        case 114: SCRAMBLED_RINV(631); break;
        case 115: SCRAMBLED_RINV(641); break;
        case 116: SCRAMBLED_RINV(643); break;
        case 117: SCRAMBLED_RINV(647); break;
        case 118: SCRAMBLED_RINV(653); break;
        case 119: SCRAMBLED_RINV(659); break;
        case 120: SCRAMBLED_RINV(661); break;
        case 121: SCRAMBLED_RINV(673); break;
        case 122: SCRAMBLED_RINV(677); break;
        case 123: SCRAMBLED_RINV(683); break;
        case 124: SCRAMBLED_RINV(691); break;
        case 125: SCRAMBLED_RINV(701); break;
        case 126: SCRAMBLED_RINV(709); break;
        case 127: SCRAMBLED_RINV(719); break;
        case 128: SCRAMBLED_RINV(727); break;
        case 129: SCRAMBLED_RINV(733); break;
        case 130: SCRAMBLED_RINV(739); break;
        case 131: SCRAMBLED_RINV(743); break;
        case 132: SCRAMBLED_RINV(751); break;
        case 133: SCRAMBLED_RINV(757); break;
        case 134: SCRAMBLED_RINV(761); break;
        case 135: SCRAMBLED_RINV(769); break;
        case 136: SCRAMBLED_RINV(773); break;
        case 137: SCRAMBLED_RINV(787); break;
        case 138: SCRAMBLED_RINV(797); break;
        case 139: SCRAMBLED_RINV(809); break;
        case 140: SCRAMBLED_RINV(811); break;
        case 141: SCRAMBLED_RINV(821); break;
        case 142: SCRAMBLED_RINV(823); break;
        case 143: SCRAMBLED_RINV(827); break;
        case 144: SCRAMBLED_RINV(829); break;
        case 145: SCRAMBLED_RINV(839); break;
        case 146: SCRAMBLED_RINV(853); break;
        case 147: SCRAMBLED_RINV(857); break;
        case 148: SCRAMBLED_RINV(859); break;
        case 149: SCRAMBLED_RINV(863); break;
        case 150: SCRAMBLED_RINV(877); break;
        case 151: SCRAMBLED_RINV(881); break;
        case 152: SCRAMBLED_RINV(883); break;
        case 153: SCRAMBLED_RINV(887); break;
        case 154: SCRAMBLED_RINV(907); break;
        case 155: SCRAMBLED_RINV(911); break;
        case 156: SCRAMBLED_RINV(919); break;
        case 157: SCRAMBLED_RINV(929); break;
        case 158: SCRAMBLED_RINV(937); break;
        case 159: SCRAMBLED_RINV(941); break;
        case 160: SCRAMBLED_RINV(947); break;
        case 161: SCRAMBLED_RINV(953); break;
        case 162: SCRAMBLED_RINV(967); break;
        case 163: SCRAMBLED_RINV(971); break;
        case 164: SCRAMBLED_RINV(977); break;
        case 165: SCRAMBLED_RINV(983); break;
        case 166: SCRAMBLED_RINV(991); break;
        case 167: SCRAMBLED_RINV(997); break;
        case 168: SCRAMBLED_RINV(1009); break;
        case 169: SCRAMBLED_RINV(1013); break;
        case 170: SCRAMBLED_RINV(1019); break;
        case 171: SCRAMBLED_RINV(1021); break;
        case 172: SCRAMBLED_RINV(1031); break;
        case 173: SCRAMBLED_RINV(1033); break;
        case 174: SCRAMBLED_RINV(1039); break;
        case 175: SCRAMBLED_RINV(1049); break;
        case 176: SCRAMBLED_RINV(1051); break;
        case 177: SCRAMBLED_RINV(1061); break;
        case 178: SCRAMBLED_RINV(1063); break;
        case 179: SCRAMBLED_RINV(1069); break;
        case 180: SCRAMBLED_RINV(1087); break;
        case 181: SCRAMBLED_RINV(1091); break;
        case 182: SCRAMBLED_RINV(1093); break;
        case 183: SCRAMBLED_RINV(1097); break;
        case 184: SCRAMBLED_RINV(1103); break;
        case 185: SCRAMBLED_RINV(1109); break;
        case 186: SCRAMBLED_RINV(1117); break;
        case 187: SCRAMBLED_RINV(1123); break;
        case 188: SCRAMBLED_RINV(1129); break;
        case 189: SCRAMBLED_RINV(1151); break;
        case 190: SCRAMBLED_RINV(1153); break;
        case 191: SCRAMBLED_RINV(1163); break;
        case 192: SCRAMBLED_RINV(1171); break;
        case 193: SCRAMBLED_RINV(1181); break;
        case 194: SCRAMBLED_RINV(1187); break;
        case 195: SCRAMBLED_RINV(1193); break;
        case 196: SCRAMBLED_RINV(1201); break;
        case 197: SCRAMBLED_RINV(1213); break;
        case 198: SCRAMBLED_RINV(1217); break;
        case 199: SCRAMBLED_RINV(1223); break;
        case 200: SCRAMBLED_RINV(1229); break;
        case 201: SCRAMBLED_RINV(1231); break;
        case 202: SCRAMBLED_RINV(1237); break;
        case 203: SCRAMBLED_RINV(1249); break;
        case 204: SCRAMBLED_RINV(1259); break;
        case 205: SCRAMBLED_RINV(1277); break;
        case 206: SCRAMBLED_RINV(1279); break;
        case 207: SCRAMBLED_RINV(1283); break;
        case 208: SCRAMBLED_RINV(1289); break;
        case 209: SCRAMBLED_RINV(1291); break;
        case 210: SCRAMBLED_RINV(1297); break;
        case 211: SCRAMBLED_RINV(1301); break;
        case 212: SCRAMBLED_RINV(1303); break;
        case 213: SCRAMBLED_RINV(1307); break;
        case 214: SCRAMBLED_RINV(1319); break;
        case 215: SCRAMBLED_RINV(1321); break;
        case 216: SCRAMBLED_RINV(1327); break;
        case 217: SCRAMBLED_RINV(1361); break;
        case 218: SCRAMBLED_RINV(1367); break;
        case 219: SCRAMBLED_RINV(1373); break;
        case 220: SCRAMBLED_RINV(1381); break;
        case 221: SCRAMBLED_RINV(1399); break;
        case 222: SCRAMBLED_RINV(1409); break;
        case 223: SCRAMBLED_RINV(1423); break;
        case 224: SCRAMBLED_RINV(1427); break;
        case 225: SCRAMBLED_RINV(1429); break;
        case 226: SCRAMBLED_RINV(1433); break;
        case 227: SCRAMBLED_RINV(1439); break;
        case 228: SCRAMBLED_RINV(1447); break;
        case 229: SCRAMBLED_RINV(1451); break;
        case 230: SCRAMBLED_RINV(1453); break;
        case 231: SCRAMBLED_RINV(1459); break;
        case 232: SCRAMBLED_RINV(1471); break;
        case 233: SCRAMBLED_RINV(1481); break;
        case 234: SCRAMBLED_RINV(1483); break;
        case 235: SCRAMBLED_RINV(1487); break;
        case 236: SCRAMBLED_RINV(1489); break;
        case 237: SCRAMBLED_RINV(1493); break;
        case 238: SCRAMBLED_RINV(1499); break;
        case 239: SCRAMBLED_RINV(1511); break;
        case 240: SCRAMBLED_RINV(1523); break;
        case 241: SCRAMBLED_RINV(1531); break;
        case 242: SCRAMBLED_RINV(1543); break;
        case 243: SCRAMBLED_RINV(1549); break;
        case 244: SCRAMBLED_RINV(1553); break;
        case 245: SCRAMBLED_RINV(1559); break;
        case 246: SCRAMBLED_RINV(1567); break;
        case 247: SCRAMBLED_RINV(1571); break;
        case 248: SCRAMBLED_RINV(1579); break;
        case 249: SCRAMBLED_RINV(1583); break;
        case 250: SCRAMBLED_RINV(1597); break;
        case 251: SCRAMBLED_RINV(1601); break;
        case 252: SCRAMBLED_RINV(1607); break;
        case 253: SCRAMBLED_RINV(1609); break;
        case 254: SCRAMBLED_RINV(1613); break;
        case 255: SCRAMBLED_RINV(1619); break;
        case 256: SCRAMBLED_RINV(1621); break;
        case 257: SCRAMBLED_RINV(1627); break;
        case 258: SCRAMBLED_RINV(1637); break;
        case 259: SCRAMBLED_RINV(1657); break;
        case 260: SCRAMBLED_RINV(1663); break;
        case 261: SCRAMBLED_RINV(1667); break;
        case 262: SCRAMBLED_RINV(1669); break;
        case 263: SCRAMBLED_RINV(1693); break;
        case 264: SCRAMBLED_RINV(1697); break;
        case 265: SCRAMBLED_RINV(1699); break;
        case 266: SCRAMBLED_RINV(1709); break;
        case 267: SCRAMBLED_RINV(1721); break;
        case 268: SCRAMBLED_RINV(1723); break;
        case 269: SCRAMBLED_RINV(1733); break;
        case 270: SCRAMBLED_RINV(1741); break;
        case 271: SCRAMBLED_RINV(1747); break;
        case 272: SCRAMBLED_RINV(1753); break;
        case 273: SCRAMBLED_RINV(1759); break;
        case 274: SCRAMBLED_RINV(1777); break;
        case 275: SCRAMBLED_RINV(1783); break;
        case 276: SCRAMBLED_RINV(1787); break;
        case 277: SCRAMBLED_RINV(1789); break;
        case 278: SCRAMBLED_RINV(1801); break;
        case 279: SCRAMBLED_RINV(1811); break;
        case 280: SCRAMBLED_RINV(1823); break;
        case 281: SCRAMBLED_RINV(1831); break;
        case 282: SCRAMBLED_RINV(1847); break;
        case 283: SCRAMBLED_RINV(1861); break;
        case 284: SCRAMBLED_RINV(1867); break;
        case 285: SCRAMBLED_RINV(1871); break;
        case 286: SCRAMBLED_RINV(1873); break;
        case 287: SCRAMBLED_RINV(1877); break;
        case 288: SCRAMBLED_RINV(1879); break;
        case 289: SCRAMBLED_RINV(1889); break;
        case 290: SCRAMBLED_RINV(1901); break;
        case 291: SCRAMBLED_RINV(1907); break;
        case 292: SCRAMBLED_RINV(1913); break;
        case 293: SCRAMBLED_RINV(1931); break;
        case 294: SCRAMBLED_RINV(1933); break;
        case 295: SCRAMBLED_RINV(1949); break;
        case 296: SCRAMBLED_RINV(1951); break;
        case 297: SCRAMBLED_RINV(1973); break;
        case 298: SCRAMBLED_RINV(1979); break;
        case 299: SCRAMBLED_RINV(1987); break;
        case 300: SCRAMBLED_RINV(1993); break;
        case 301: SCRAMBLED_RINV(1997); break;
        case 302: SCRAMBLED_RINV(1999); break;
        case 303: SCRAMBLED_RINV(2003); break;
        case 304: SCRAMBLED_RINV(2011); break;
        case 305: SCRAMBLED_RINV(2017); break;
        case 306: SCRAMBLED_RINV(2027); break;
        case 307: SCRAMBLED_RINV(2029); break;
        case 308: SCRAMBLED_RINV(2039); break;
        case 309: SCRAMBLED_RINV(2053); break;
        case 310: SCRAMBLED_RINV(2063); break;
        case 311: SCRAMBLED_RINV(2069); break;
        case 312: SCRAMBLED_RINV(2081); break;
        case 313: SCRAMBLED_RINV(2083); break;
        case 314: SCRAMBLED_RINV(2087); break;
        case 315: SCRAMBLED_RINV(2089); break;
        case 316: SCRAMBLED_RINV(2099); break;
        case 317: SCRAMBLED_RINV(2111); break;
        case 318: SCRAMBLED_RINV(2113); break;
        case 319: SCRAMBLED_RINV(2129); break;
        case 320: SCRAMBLED_RINV(2131); break;
        case 321: SCRAMBLED_RINV(2137); break;
        case 322: SCRAMBLED_RINV(2141); break;
        case 323: SCRAMBLED_RINV(2143); break;
        case 324: SCRAMBLED_RINV(2153); break;
        case 325: SCRAMBLED_RINV(2161); break;
        case 326: SCRAMBLED_RINV(2179); break;
        case 327: SCRAMBLED_RINV(2203); break;
        case 328: SCRAMBLED_RINV(2207); break;
        case 329: SCRAMBLED_RINV(2213); break;
        case 330: SCRAMBLED_RINV(2221); break;
        case 331: SCRAMBLED_RINV(2237); break;
        case 332: SCRAMBLED_RINV(2239); break;
        case 333: SCRAMBLED_RINV(2243); break;
        case 334: SCRAMBLED_RINV(2251); break;
        case 335: SCRAMBLED_RINV(2267); break;
        case 336: SCRAMBLED_RINV(2269); break;
        case 337: SCRAMBLED_RINV(2273); break;
        case 338: SCRAMBLED_RINV(2281); break;
        case 339: SCRAMBLED_RINV(2287); break;
        case 340: SCRAMBLED_RINV(2293); break;
        case 341: SCRAMBLED_RINV(2297); break;
        case 342: SCRAMBLED_RINV(2309); break;
        case 343: SCRAMBLED_RINV(2311); break;
        case 344: SCRAMBLED_RINV(2333); break;
        case 345: SCRAMBLED_RINV(2339); break;
        case 346: SCRAMBLED_RINV(2341); break;
        case 347: SCRAMBLED_RINV(2347); break;
        case 348: SCRAMBLED_RINV(2351); break;
        case 349: SCRAMBLED_RINV(2357); break;
        case 350: SCRAMBLED_RINV(2371); break;
        case 351: SCRAMBLED_RINV(2377); break;
        case 352: SCRAMBLED_RINV(2381); break;
        case 353: SCRAMBLED_RINV(2383); break;
        case 354: SCRAMBLED_RINV(2389); break;
        case 355: SCRAMBLED_RINV(2393); break;
        case 356: SCRAMBLED_RINV(2399); break;
        case 357: SCRAMBLED_RINV(2411); break;
        case 358: SCRAMBLED_RINV(2417); break;
        case 359: SCRAMBLED_RINV(2423); break;
        case 360: SCRAMBLED_RINV(2437); break;
        case 361: SCRAMBLED_RINV(2441); break;
        case 362: SCRAMBLED_RINV(2447); break;
        case 363: SCRAMBLED_RINV(2459); break;
        case 364: SCRAMBLED_RINV(2467); break;
        case 365: SCRAMBLED_RINV(2473); break;
        case 366: SCRAMBLED_RINV(2477); break;
        case 367: SCRAMBLED_RINV(2503); break;
        case 368: SCRAMBLED_RINV(2521); break;
        case 369: SCRAMBLED_RINV(2531); break;
        case 370: SCRAMBLED_RINV(2539); break;
        case 371: SCRAMBLED_RINV(2543); break;
        case 372: SCRAMBLED_RINV(2549); break;
        case 373: SCRAMBLED_RINV(2551); break;
        case 374: SCRAMBLED_RINV(2557); break;
        case 375: SCRAMBLED_RINV(2579); break;
        case 376: SCRAMBLED_RINV(2591); break;
        case 377: SCRAMBLED_RINV(2593); break;
        case 378: SCRAMBLED_RINV(2609); break;
        case 379: SCRAMBLED_RINV(2617); break;
        case 380: SCRAMBLED_RINV(2621); break;
        case 381: SCRAMBLED_RINV(2633); break;
        case 382: SCRAMBLED_RINV(2647); break;
        case 383: SCRAMBLED_RINV(2657); break;
        case 384: SCRAMBLED_RINV(2659); break;
        case 385: SCRAMBLED_RINV(2663); break;
        case 386: SCRAMBLED_RINV(2671); break;
        case 387: SCRAMBLED_RINV(2677); break;
        case 388: SCRAMBLED_RINV(2683); break;
        case 389: SCRAMBLED_RINV(2687); break;
        case 390: SCRAMBLED_RINV(2689); break;
        case 391: SCRAMBLED_RINV(2693); break;
        case 392: SCRAMBLED_RINV(2699); break;
        case 393: SCRAMBLED_RINV(2707); break;
        case 394: SCRAMBLED_RINV(2711); break;
        case 395: SCRAMBLED_RINV(2713); break;
        case 396: SCRAMBLED_RINV(2719); break;
        case 397: SCRAMBLED_RINV(2729); break;
        case 398: SCRAMBLED_RINV(2731); break;
        case 399: SCRAMBLED_RINV(2741); break;
        case 400: SCRAMBLED_RINV(2749); break;
        case 401: SCRAMBLED_RINV(2753); break;
        case 402: SCRAMBLED_RINV(2767); break;
        case 403: SCRAMBLED_RINV(2777); break;
        case 404: SCRAMBLED_RINV(2789); break;
        case 405: SCRAMBLED_RINV(2791); break;
        case 406: SCRAMBLED_RINV(2797); break;
        case 407: SCRAMBLED_RINV(2801); break;
        case 408: SCRAMBLED_RINV(2803); break;
        case 409: SCRAMBLED_RINV(2819); break;
        case 410: SCRAMBLED_RINV(2833); break;
        case 411: SCRAMBLED_RINV(2837); break;
        case 412: SCRAMBLED_RINV(2843); break;
        case 413: SCRAMBLED_RINV(2851); break;
        case 414: SCRAMBLED_RINV(2857); break;
        case 415: SCRAMBLED_RINV(2861); break;
        case 416: SCRAMBLED_RINV(2879); break;
        case 417: SCRAMBLED_RINV(2887); break;
        case 418: SCRAMBLED_RINV(2897); break;
        case 419: SCRAMBLED_RINV(2903); break;
        case 420: SCRAMBLED_RINV(2909); break;
        case 421: SCRAMBLED_RINV(2917); break;
        case 422: SCRAMBLED_RINV(2927); break;
        case 423: SCRAMBLED_RINV(2939); break;
        case 424: SCRAMBLED_RINV(2953); break;
        case 425: SCRAMBLED_RINV(2957); break;
        case 426: SCRAMBLED_RINV(2963); break;
        case 427: SCRAMBLED_RINV(2969); break;
        case 428: SCRAMBLED_RINV(2971); break;
        case 429: SCRAMBLED_RINV(2999); break;
        case 430: SCRAMBLED_RINV(3001); break;
        case 431: SCRAMBLED_RINV(3011); break;
        case 432: SCRAMBLED_RINV(3019); break;
        case 433: SCRAMBLED_RINV(3023); break;
        case 434: SCRAMBLED_RINV(3037); break;
        case 435: SCRAMBLED_RINV(3041); break;
        case 436: SCRAMBLED_RINV(3049); break;
        case 437: SCRAMBLED_RINV(3061); break;
        case 438: SCRAMBLED_RINV(3067); break;
        case 439: SCRAMBLED_RINV(3079); break;
        case 440: SCRAMBLED_RINV(3083); break;
        case 441: SCRAMBLED_RINV(3089); break;
        case 442: SCRAMBLED_RINV(3109); break;
        case 443: SCRAMBLED_RINV(3119); break;
        case 444: SCRAMBLED_RINV(3121); break;
        case 445: SCRAMBLED_RINV(3137); break;
        case 446: SCRAMBLED_RINV(3163); break;
        case 447: SCRAMBLED_RINV(3167); break;
        case 448: SCRAMBLED_RINV(3169); break;
        case 449: SCRAMBLED_RINV(3181); break;
        case 450: SCRAMBLED_RINV(3187); break;
        case 451: SCRAMBLED_RINV(3191); break;
        case 452: SCRAMBLED_RINV(3203); break;
        case 453: SCRAMBLED_RINV(3209); break;
        case 454: SCRAMBLED_RINV(3217); break;
        case 455: SCRAMBLED_RINV(3221); break;
        case 456: SCRAMBLED_RINV(3229); break;
        case 457: SCRAMBLED_RINV(3251); break;
        case 458: SCRAMBLED_RINV(3253); break;
        case 459: SCRAMBLED_RINV(3257); break;
        case 460: SCRAMBLED_RINV(3259); break;
        case 461: SCRAMBLED_RINV(3271); break;
        case 462: SCRAMBLED_RINV(3299); break;
        case 463: SCRAMBLED_RINV(3301); break;
        case 464: SCRAMBLED_RINV(3307); break;
        case 465: SCRAMBLED_RINV(3313); break;
        case 466: SCRAMBLED_RINV(3319); break;
        case 467: SCRAMBLED_RINV(3323); break;
        case 468: SCRAMBLED_RINV(3329); break;
        case 469: SCRAMBLED_RINV(3331); break;
        case 470: SCRAMBLED_RINV(3343); break;
        case 471: SCRAMBLED_RINV(3347); break;
        case 472: SCRAMBLED_RINV(3359); break;
        case 473: SCRAMBLED_RINV(3361); break;
        case 474: SCRAMBLED_RINV(3371); break;
        case 475: SCRAMBLED_RINV(3373); break;
        case 476: SCRAMBLED_RINV(3389); break;
        case 477: SCRAMBLED_RINV(3391); break;
        case 478: SCRAMBLED_RINV(3407); break;
        case 479: SCRAMBLED_RINV(3413); break;
        case 480: SCRAMBLED_RINV(3433); break;
        case 481: SCRAMBLED_RINV(3449); break;
        case 482: SCRAMBLED_RINV(3457); break;
        case 483: SCRAMBLED_RINV(3461); break;
        case 484: SCRAMBLED_RINV(3463); break;
        case 485: SCRAMBLED_RINV(3467); break;
        case 486: SCRAMBLED_RINV(3469); break;
        case 487: SCRAMBLED_RINV(3491); break;
        case 488: SCRAMBLED_RINV(3499); break;
        case 489: SCRAMBLED_RINV(3511); break;
        case 490: SCRAMBLED_RINV(3517); break;
        case 491: SCRAMBLED_RINV(3527); break;
        case 492: SCRAMBLED_RINV(3529); break;
        case 493: SCRAMBLED_RINV(3533); break;
        case 494: SCRAMBLED_RINV(3539); break;
        case 495: SCRAMBLED_RINV(3541); break;
        case 496: SCRAMBLED_RINV(3547); break;
        case 497: SCRAMBLED_RINV(3557); break;
        case 498: SCRAMBLED_RINV(3559); break;
        case 499: SCRAMBLED_RINV(3571); break;
        case 500: SCRAMBLED_RINV(3581); break;
        case 501: SCRAMBLED_RINV(3583); break;
        case 502: SCRAMBLED_RINV(3593); break;
        case 503: SCRAMBLED_RINV(3607); break;
        case 504: SCRAMBLED_RINV(3613); break;
        case 505: SCRAMBLED_RINV(3617); break;
        case 506: SCRAMBLED_RINV(3623); break;
        case 507: SCRAMBLED_RINV(3631); break;
        case 508: SCRAMBLED_RINV(3637); break;
        case 509: SCRAMBLED_RINV(3643); break;
        case 510: SCRAMBLED_RINV(3659); break;
        case 511: SCRAMBLED_RINV(3671); break;
        case 512: SCRAMBLED_RINV(3673); break;
        case 513: SCRAMBLED_RINV(3677); break;
        case 514: SCRAMBLED_RINV(3691); break;
        case 515: SCRAMBLED_RINV(3697); break;
        case 516: SCRAMBLED_RINV(3701); break;
        case 517: SCRAMBLED_RINV(3709); break;
        case 518: SCRAMBLED_RINV(3719); break;
        case 519: SCRAMBLED_RINV(3727); break;
        case 520: SCRAMBLED_RINV(3733); break;
        case 521: SCRAMBLED_RINV(3739); break;
        case 522: SCRAMBLED_RINV(3761); break;
        case 523: SCRAMBLED_RINV(3767); break;
        case 524: SCRAMBLED_RINV(3769); break;
        case 525: SCRAMBLED_RINV(3779); break;
        case 526: SCRAMBLED_RINV(3793); break;
        case 527: SCRAMBLED_RINV(3797); break;
        case 528: SCRAMBLED_RINV(3803); break;
        case 529: SCRAMBLED_RINV(3821); break;
        case 530: SCRAMBLED_RINV(3823); break;
        case 531: SCRAMBLED_RINV(3833); break;
        case 532: SCRAMBLED_RINV(3847); break;
        case 533: SCRAMBLED_RINV(3851); break;
        case 534: SCRAMBLED_RINV(3853); break;
        case 535: SCRAMBLED_RINV(3863); break;
        case 536: SCRAMBLED_RINV(3877); break;
        case 537: SCRAMBLED_RINV(3881); break;
        case 538: SCRAMBLED_RINV(3889); break;
        case 539: SCRAMBLED_RINV(3907); break;
        case 540: SCRAMBLED_RINV(3911); break;
        case 541: SCRAMBLED_RINV(3917); break;
        case 542: SCRAMBLED_RINV(3919); break;
        case 543: SCRAMBLED_RINV(3923); break;
        case 544: SCRAMBLED_RINV(3929); break;
        case 545: SCRAMBLED_RINV(3931); break;
        case 546: SCRAMBLED_RINV(3943); break;
        case 547: SCRAMBLED_RINV(3947); break;
        case 548: SCRAMBLED_RINV(3967); break;
        case 549: SCRAMBLED_RINV(3989); break;
        case 550: SCRAMBLED_RINV(4001); break;
        case 551: SCRAMBLED_RINV(4003); break;
        case 552: SCRAMBLED_RINV(4007); break;
        case 553: SCRAMBLED_RINV(4013); break;
        case 554: SCRAMBLED_RINV(4019); break;
        case 555: SCRAMBLED_RINV(4021); break;
        case 556: SCRAMBLED_RINV(4027); break;
        case 557: SCRAMBLED_RINV(4049); break;
        case 558: SCRAMBLED_RINV(4051); break;
        case 559: SCRAMBLED_RINV(4057); break;
        case 560: SCRAMBLED_RINV(4073); break;
        case 561: SCRAMBLED_RINV(4079); break;
        case 562: SCRAMBLED_RINV(4091); break;
        case 563: SCRAMBLED_RINV(4093); break;
        case 564: SCRAMBLED_RINV(4099); break;
        case 565: SCRAMBLED_RINV(4111); break;
        case 566: SCRAMBLED_RINV(4127); break;
        case 567: SCRAMBLED_RINV(4129); break;
        case 568: SCRAMBLED_RINV(4133); break;
        case 569: SCRAMBLED_RINV(4139); break;
        case 570: SCRAMBLED_RINV(4153); break;
        case 571: SCRAMBLED_RINV(4157); break;
        case 572: SCRAMBLED_RINV(4159); break;
        case 573: SCRAMBLED_RINV(4177); break;
        case 574: SCRAMBLED_RINV(4201); break;
        case 575: SCRAMBLED_RINV(4211); break;
        case 576: SCRAMBLED_RINV(4217); break;
        case 577: SCRAMBLED_RINV(4219); break;
        case 578: SCRAMBLED_RINV(4229); break;
        case 579: SCRAMBLED_RINV(4231); break;
        case 580: SCRAMBLED_RINV(4241); break;
        case 581: SCRAMBLED_RINV(4243); break;
        case 582: SCRAMBLED_RINV(4253); break;
        case 583: SCRAMBLED_RINV(4259); break;
        case 584: SCRAMBLED_RINV(4261); break;
        case 585: SCRAMBLED_RINV(4271); break;
        case 586: SCRAMBLED_RINV(4273); break;
        case 587: SCRAMBLED_RINV(4283); break;
        case 588: SCRAMBLED_RINV(4289); break;
        case 589: SCRAMBLED_RINV(4297); break;
        case 590: SCRAMBLED_RINV(4327); break;
        case 591: SCRAMBLED_RINV(4337); break;
        case 592: SCRAMBLED_RINV(4339); break;
        case 593: SCRAMBLED_RINV(4349); break;
        case 594: SCRAMBLED_RINV(4357); break;
        case 595: SCRAMBLED_RINV(4363); break;
        case 596: SCRAMBLED_RINV(4373); break;
        case 597: SCRAMBLED_RINV(4391); break;
        case 598: SCRAMBLED_RINV(4397); break;
        case 599: SCRAMBLED_RINV(4409); break;
        case 600: SCRAMBLED_RINV(4421); break;
        case 601: SCRAMBLED_RINV(4423); break;
        case 602: SCRAMBLED_RINV(4441); break;
        case 603: SCRAMBLED_RINV(4447); break;
        case 604: SCRAMBLED_RINV(4451); break;
        case 605: SCRAMBLED_RINV(4457); break;
        case 606: SCRAMBLED_RINV(4463); break;
        case 607: SCRAMBLED_RINV(4481); break;
        case 608: SCRAMBLED_RINV(4483); break;
        case 609: SCRAMBLED_RINV(4493); break;
        case 610: SCRAMBLED_RINV(4507); break;
        case 611: SCRAMBLED_RINV(4513); break;
        case 612: SCRAMBLED_RINV(4517); break;
        case 613: SCRAMBLED_RINV(4519); break;
        case 614: SCRAMBLED_RINV(4523); break;
        case 615: SCRAMBLED_RINV(4547); break;
        case 616: SCRAMBLED_RINV(4549); break;
        case 617: SCRAMBLED_RINV(4561); break;
        case 618: SCRAMBLED_RINV(4567); break;
        case 619: SCRAMBLED_RINV(4583); break;
        case 620: SCRAMBLED_RINV(4591); break;
        case 621: SCRAMBLED_RINV(4597); break;
        case 622: SCRAMBLED_RINV(4603); break;
        case 623: SCRAMBLED_RINV(4621); break;
        case 624: SCRAMBLED_RINV(4637); break;
        case 625: SCRAMBLED_RINV(4639); break;
        case 626: SCRAMBLED_RINV(4643); break;
        case 627: SCRAMBLED_RINV(4649); break;
        case 628: SCRAMBLED_RINV(4651); break;
        case 629: SCRAMBLED_RINV(4657); break;
        case 630: SCRAMBLED_RINV(4663); break;
        case 631: SCRAMBLED_RINV(4673); break;
        case 632: SCRAMBLED_RINV(4679); break;
        case 633: SCRAMBLED_RINV(4691); break;
        case 634: SCRAMBLED_RINV(4703); break;
        case 635: SCRAMBLED_RINV(4721); break;
        case 636: SCRAMBLED_RINV(4723); break;
        case 637: SCRAMBLED_RINV(4729); break;
        case 638: SCRAMBLED_RINV(4733); break;
        case 639: SCRAMBLED_RINV(4751); break;
        case 640: SCRAMBLED_RINV(4759); break;
        case 641: SCRAMBLED_RINV(4783); break;
        case 642: SCRAMBLED_RINV(4787); break;
        case 643: SCRAMBLED_RINV(4789); break;
        case 644: SCRAMBLED_RINV(4793); break;
        case 645: SCRAMBLED_RINV(4799); break;
        case 646: SCRAMBLED_RINV(4801); break;
        case 647: SCRAMBLED_RINV(4813); break;
        case 648: SCRAMBLED_RINV(4817); break;
        case 649: SCRAMBLED_RINV(4831); break;
        case 650: SCRAMBLED_RINV(4861); break;
        case 651: SCRAMBLED_RINV(4871); break;
        case 652: SCRAMBLED_RINV(4877); break;
        case 653: SCRAMBLED_RINV(4889); break;
        case 654: SCRAMBLED_RINV(4903); break;
        case 655: SCRAMBLED_RINV(4909); break;
        case 656: SCRAMBLED_RINV(4919); break;
        case 657: SCRAMBLED_RINV(4931); break;
        case 658: SCRAMBLED_RINV(4933); break;
        case 659: SCRAMBLED_RINV(4937); break;
        case 660: SCRAMBLED_RINV(4943); break;
        case 661: SCRAMBLED_RINV(4951); break;
        case 662: SCRAMBLED_RINV(4957); break;
        case 663: SCRAMBLED_RINV(4967); break;
        case 664: SCRAMBLED_RINV(4969); break;
        case 665: SCRAMBLED_RINV(4973); break;
        case 666: SCRAMBLED_RINV(4987); break;
        case 667: SCRAMBLED_RINV(4993); break;
        case 668: SCRAMBLED_RINV(4999); break;
        case 669: SCRAMBLED_RINV(5003); break;
        case 670: SCRAMBLED_RINV(5009); break;
        case 671: SCRAMBLED_RINV(5011); break;
        case 672: SCRAMBLED_RINV(5021); break;
        case 673: SCRAMBLED_RINV(5023); break;
        case 674: SCRAMBLED_RINV(5039); break;
        case 675: SCRAMBLED_RINV(5051); break;
        case 676: SCRAMBLED_RINV(5059); break;
        case 677: SCRAMBLED_RINV(5077); break;
        case 678: SCRAMBLED_RINV(5081); break;
        case 679: SCRAMBLED_RINV(5087); break;
        case 680: SCRAMBLED_RINV(5099); break;
        case 681: SCRAMBLED_RINV(5101); break;
        case 682: SCRAMBLED_RINV(5107); break;
        case 683: SCRAMBLED_RINV(5113); break;
        case 684: SCRAMBLED_RINV(5119); break;
        case 685: SCRAMBLED_RINV(5147); break;
        case 686: SCRAMBLED_RINV(5153); break;
        case 687: SCRAMBLED_RINV(5167); break;
        case 688: SCRAMBLED_RINV(5171); break;
        case 689: SCRAMBLED_RINV(5179); break;
        case 690: SCRAMBLED_RINV(5189); break;
        case 691: SCRAMBLED_RINV(5197); break;
        case 692: SCRAMBLED_RINV(5209); break;
        case 693: SCRAMBLED_RINV(5227); break;
        case 694: SCRAMBLED_RINV(5231); break;
        case 695: SCRAMBLED_RINV(5233); break;
        case 696: SCRAMBLED_RINV(5237); break;
        case 697: SCRAMBLED_RINV(5261); break;
        case 698: SCRAMBLED_RINV(5273); break;
        case 699: SCRAMBLED_RINV(5279); break;
        case 700: SCRAMBLED_RINV(5281); break;
        case 701: SCRAMBLED_RINV(5297); break;
        case 702: SCRAMBLED_RINV(5303); break;
        case 703: SCRAMBLED_RINV(5309); break;
        case 704: SCRAMBLED_RINV(5323); break;
        case 705: SCRAMBLED_RINV(5333); break;
        case 706: SCRAMBLED_RINV(5347); break;
        case 707: SCRAMBLED_RINV(5351); break;
        case 708: SCRAMBLED_RINV(5381); break;
        case 709: SCRAMBLED_RINV(5387); break;
        case 710: SCRAMBLED_RINV(5393); break;
        case 711: SCRAMBLED_RINV(5399); break;
        case 712: SCRAMBLED_RINV(5407); break;
        case 713: SCRAMBLED_RINV(5413); break;
        case 714: SCRAMBLED_RINV(5417); break;
        case 715: SCRAMBLED_RINV(5419); break;
        case 716: SCRAMBLED_RINV(5431); break;
        case 717: SCRAMBLED_RINV(5437); break;
        case 718: SCRAMBLED_RINV(5441); break;
        case 719: SCRAMBLED_RINV(5443); break;
        case 720: SCRAMBLED_RINV(5449); break;
        case 721: SCRAMBLED_RINV(5471); break;
        case 722: SCRAMBLED_RINV(5477); break;
        case 723: SCRAMBLED_RINV(5479); break;
        case 724: SCRAMBLED_RINV(5483); break;
        case 725: SCRAMBLED_RINV(5501); break;
        case 726: SCRAMBLED_RINV(5503); break;
        case 727: SCRAMBLED_RINV(5507); break;
        case 728: SCRAMBLED_RINV(5519); break;
        case 729: SCRAMBLED_RINV(5521); break;
        case 730: SCRAMBLED_RINV(5527); break;
        case 731: SCRAMBLED_RINV(5531); break;
        case 732: SCRAMBLED_RINV(5557); break;
        case 733: SCRAMBLED_RINV(5563); break;
        case 734: SCRAMBLED_RINV(5569); break;
        case 735: SCRAMBLED_RINV(5573); break;
        case 736: SCRAMBLED_RINV(5581); break;
        case 737: SCRAMBLED_RINV(5591); break;
        case 738: SCRAMBLED_RINV(5623); break;
        case 739: SCRAMBLED_RINV(5639); break;
        case 740: SCRAMBLED_RINV(5641); break;
        case 741: SCRAMBLED_RINV(5647); break;
        case 742: SCRAMBLED_RINV(5651); break;
        case 743: SCRAMBLED_RINV(5653); break;
        case 744: SCRAMBLED_RINV(5657); break;
        case 745: SCRAMBLED_RINV(5659); break;
        case 746: SCRAMBLED_RINV(5669); break;
        case 747: SCRAMBLED_RINV(5683); break;
        case 748: SCRAMBLED_RINV(5689); break;
        case 749: SCRAMBLED_RINV(5693); break;
        case 750: SCRAMBLED_RINV(5701); break;
        case 751: SCRAMBLED_RINV(5711); break;
        case 752: SCRAMBLED_RINV(5717); break;
        case 753: SCRAMBLED_RINV(5737); break;
        case 754: SCRAMBLED_RINV(5741); break;
        case 755: SCRAMBLED_RINV(5743); break;
        case 756: SCRAMBLED_RINV(5749); break;
        case 757: SCRAMBLED_RINV(5779); break;
        case 758: SCRAMBLED_RINV(5783); break;
        case 759: SCRAMBLED_RINV(5791); break;
        case 760: SCRAMBLED_RINV(5801); break;
        case 761: SCRAMBLED_RINV(5807); break;
        case 762: SCRAMBLED_RINV(5813); break;
        case 763: SCRAMBLED_RINV(5821); break;
        case 764: SCRAMBLED_RINV(5827); break;
        case 765: SCRAMBLED_RINV(5839); break;
        case 766: SCRAMBLED_RINV(5843); break;
        case 767: SCRAMBLED_RINV(5849); break;
        case 768: SCRAMBLED_RINV(5851); break;
        case 769: SCRAMBLED_RINV(5857); break;
        case 770: SCRAMBLED_RINV(5861); break;
        case 771: SCRAMBLED_RINV(5867); break;
        case 772: SCRAMBLED_RINV(5869); break;
        case 773: SCRAMBLED_RINV(5879); break;
        case 774: SCRAMBLED_RINV(5881); break;
        case 775: SCRAMBLED_RINV(5897); break;
        case 776: SCRAMBLED_RINV(5903); break;
        case 777: SCRAMBLED_RINV(5923); break;
        case 778: SCRAMBLED_RINV(5927); break;
        case 779: SCRAMBLED_RINV(5939); break;
        case 780: SCRAMBLED_RINV(5953); break;
        case 781: SCRAMBLED_RINV(5981); break;
        case 782: SCRAMBLED_RINV(5987); break;
        case 783: SCRAMBLED_RINV(6007); break;
        case 784: SCRAMBLED_RINV(6011); break;
        case 785: SCRAMBLED_RINV(6029); break;
        case 786: SCRAMBLED_RINV(6037); break;
        case 787: SCRAMBLED_RINV(6043); break;
        case 788: SCRAMBLED_RINV(6047); break;
        case 789: SCRAMBLED_RINV(6053); break;
        case 790: SCRAMBLED_RINV(6067); break;
        case 791: SCRAMBLED_RINV(6073); break;
        case 792: SCRAMBLED_RINV(6079); break;
        case 793: SCRAMBLED_RINV(6089); break;
        case 794: SCRAMBLED_RINV(6091); break;
        case 795: SCRAMBLED_RINV(6101); break;
        case 796: SCRAMBLED_RINV(6113); break;
        case 797: SCRAMBLED_RINV(6121); break;
        case 798: SCRAMBLED_RINV(6131); break;
        case 799: SCRAMBLED_RINV(6133); break;
        case 800: SCRAMBLED_RINV(6143); break;
        case 801: SCRAMBLED_RINV(6151); break;
        case 802: SCRAMBLED_RINV(6163); break;
        case 803: SCRAMBLED_RINV(6173); break;
        case 804: SCRAMBLED_RINV(6197); break;
        case 805: SCRAMBLED_RINV(6199); break;
        case 806: SCRAMBLED_RINV(6203); break;
        case 807: SCRAMBLED_RINV(6211); break;
        case 808: SCRAMBLED_RINV(6217); break;
        case 809: SCRAMBLED_RINV(6221); break;
        case 810: SCRAMBLED_RINV(6229); break;
        case 811: SCRAMBLED_RINV(6247); break;
        case 812: SCRAMBLED_RINV(6257); break;
        case 813: SCRAMBLED_RINV(6263); break;
        case 814: SCRAMBLED_RINV(6269); break;
        case 815: SCRAMBLED_RINV(6271); break;
        case 816: SCRAMBLED_RINV(6277); break;
        case 817: SCRAMBLED_RINV(6287); break;
        case 818: SCRAMBLED_RINV(6299); break;
        case 819: SCRAMBLED_RINV(6301); break;
        case 820: SCRAMBLED_RINV(6311); break;
        case 821: SCRAMBLED_RINV(6317); break;
        case 822: SCRAMBLED_RINV(6323); break;
        case 823: SCRAMBLED_RINV(6329); break;
        case 824: SCRAMBLED_RINV(6337); break;
        case 825: SCRAMBLED_RINV(6343); break;
        case 826: SCRAMBLED_RINV(6353); break;
        case 827: SCRAMBLED_RINV(6359); break;
        case 828: SCRAMBLED_RINV(6361); break;
        case 829: SCRAMBLED_RINV(6367); break;
        case 830: SCRAMBLED_RINV(6373); break;
        case 831: SCRAMBLED_RINV(6379); break;
        case 832: SCRAMBLED_RINV(6389); break;
        case 833: SCRAMBLED_RINV(6397); break;
        case 834: SCRAMBLED_RINV(6421); break;
        case 835: SCRAMBLED_RINV(6427); break;
        case 836: SCRAMBLED_RINV(6449); break;
        case 837: SCRAMBLED_RINV(6451); break;
        case 838: SCRAMBLED_RINV(6469); break;
        case 839: SCRAMBLED_RINV(6473); break;
        case 840: SCRAMBLED_RINV(6481); break;
        case 841: SCRAMBLED_RINV(6491); break;
        case 842: SCRAMBLED_RINV(6521); break;
        case 843: SCRAMBLED_RINV(6529); break;
        case 844: SCRAMBLED_RINV(6547); break;
        case 845: SCRAMBLED_RINV(6551); break;
        case 846: SCRAMBLED_RINV(6553); break;
        case 847: SCRAMBLED_RINV(6563); break;
        case 848: SCRAMBLED_RINV(6569); break;
        case 849: SCRAMBLED_RINV(6571); break;
        case 850: SCRAMBLED_RINV(6577); break;
        case 851: SCRAMBLED_RINV(6581); break;
        case 852: SCRAMBLED_RINV(6599); break;
        case 853: SCRAMBLED_RINV(6607); break;
        case 854: SCRAMBLED_RINV(6619); break;
        case 855: SCRAMBLED_RINV(6637); break;
        case 856: SCRAMBLED_RINV(6653); break;
        case 857: SCRAMBLED_RINV(6659); break;
        case 858: SCRAMBLED_RINV(6661); break;
        case 859: SCRAMBLED_RINV(6673); break;
        case 860: SCRAMBLED_RINV(6679); break;
        case 861: SCRAMBLED_RINV(6689); break;
        case 862: SCRAMBLED_RINV(6691); break;
        case 863: SCRAMBLED_RINV(6701); break;
        case 864: SCRAMBLED_RINV(6703); break;
        case 865: SCRAMBLED_RINV(6709); break;
        case 866: SCRAMBLED_RINV(6719); break;
        case 867: SCRAMBLED_RINV(6733); break;
        case 868: SCRAMBLED_RINV(6737); break;
        case 869: SCRAMBLED_RINV(6761); break;
        case 870: SCRAMBLED_RINV(6763); break;
        case 871: SCRAMBLED_RINV(6779); break;
        case 872: SCRAMBLED_RINV(6781); break;
        case 873: SCRAMBLED_RINV(6791); break;
        case 874: SCRAMBLED_RINV(6793); break;
        case 875: SCRAMBLED_RINV(6803); break;
        case 876: SCRAMBLED_RINV(6823); break;
        case 877: SCRAMBLED_RINV(6827); break;
        case 878: SCRAMBLED_RINV(6829); break;
        case 879: SCRAMBLED_RINV(6833); break;
        case 880: SCRAMBLED_RINV(6841); break;
        case 881: SCRAMBLED_RINV(6857); break;
        case 882: SCRAMBLED_RINV(6863); break;
        case 883: SCRAMBLED_RINV(6869); break;
        case 884: SCRAMBLED_RINV(6871); break;
        case 885: SCRAMBLED_RINV(6883); break;
        case 886: SCRAMBLED_RINV(6899); break;
        case 887: SCRAMBLED_RINV(6907); break;
        case 888: SCRAMBLED_RINV(6911); break;
        case 889: SCRAMBLED_RINV(6917); break;
        case 890: SCRAMBLED_RINV(6947); break;
        case 891: SCRAMBLED_RINV(6949); break;
        case 892: SCRAMBLED_RINV(6959); break;
        case 893: SCRAMBLED_RINV(6961); break;
        case 894: SCRAMBLED_RINV(6967); break;
        case 895: SCRAMBLED_RINV(6971); break;
        case 896: SCRAMBLED_RINV(6977); break;
        case 897: SCRAMBLED_RINV(6983); break;
        case 898: SCRAMBLED_RINV(6991); break;
        case 899: SCRAMBLED_RINV(6997); break;
        case 900: SCRAMBLED_RINV(7001); break;
        case 901: SCRAMBLED_RINV(7013); break;
        case 902: SCRAMBLED_RINV(7019); break;
        case 903: SCRAMBLED_RINV(7027); break;
        case 904: SCRAMBLED_RINV(7039); break;
        case 905: SCRAMBLED_RINV(7043); break;
        case 906: SCRAMBLED_RINV(7057); break;
        case 907: SCRAMBLED_RINV(7069); break;
        case 908: SCRAMBLED_RINV(7079); break;
        case 909: SCRAMBLED_RINV(7103); break;
        case 910: SCRAMBLED_RINV(7109); break;
        case 911: SCRAMBLED_RINV(7121); break;
        case 912: SCRAMBLED_RINV(7127); break;
        case 913: SCRAMBLED_RINV(7129); break;
        case 914: SCRAMBLED_RINV(7151); break;
        case 915: SCRAMBLED_RINV(7159); break;
        case 916: SCRAMBLED_RINV(7177); break;
        case 917: SCRAMBLED_RINV(7187); break;
        case 918: SCRAMBLED_RINV(7193); break;
        case 919: SCRAMBLED_RINV(7207); break;
        case 920: SCRAMBLED_RINV(7211); break;
        case 921: SCRAMBLED_RINV(7213); break;
        case 922: SCRAMBLED_RINV(7219); break;
        case 923: SCRAMBLED_RINV(7229); break;
        case 924: SCRAMBLED_RINV(7237); break;
        case 925: SCRAMBLED_RINV(7243); break;
        case 926: SCRAMBLED_RINV(7247); break;
        case 927: SCRAMBLED_RINV(7253); break;
        case 928: SCRAMBLED_RINV(7283); break;
        case 929: SCRAMBLED_RINV(7297); break;
        case 930: SCRAMBLED_RINV(7307); break;
        case 931: SCRAMBLED_RINV(7309); break;
        case 932: SCRAMBLED_RINV(7321); break;
        case 933: SCRAMBLED_RINV(7331); break;
        case 934: SCRAMBLED_RINV(7333); break;
        case 935: SCRAMBLED_RINV(7349); break;
        case 936: SCRAMBLED_RINV(7351); break;
        case 937: SCRAMBLED_RINV(7369); break;
        case 938: SCRAMBLED_RINV(7393); break;
        case 939: SCRAMBLED_RINV(7411); break;
        case 940: SCRAMBLED_RINV(7417); break;
        case 941: SCRAMBLED_RINV(7433); break;
        case 942: SCRAMBLED_RINV(7451); break;
        case 943: SCRAMBLED_RINV(7457); break;
        case 944: SCRAMBLED_RINV(7459); break;
        case 945: SCRAMBLED_RINV(7477); break;
        case 946: SCRAMBLED_RINV(7481); break;
        case 947: SCRAMBLED_RINV(7487); break;
        case 948: SCRAMBLED_RINV(7489); break;
        case 949: SCRAMBLED_RINV(7499); break;
        case 950: SCRAMBLED_RINV(7507); break;
        case 951: SCRAMBLED_RINV(7517); break;
        case 952: SCRAMBLED_RINV(7523); break;
        case 953: SCRAMBLED_RINV(7529); break;
        case 954: SCRAMBLED_RINV(7537); break;
        case 955: SCRAMBLED_RINV(7541); break;
        case 956: SCRAMBLED_RINV(7547); break;
        case 957: SCRAMBLED_RINV(7549); break;
        case 958: SCRAMBLED_RINV(7559); break;
        case 959: SCRAMBLED_RINV(7561); break;
        case 960: SCRAMBLED_RINV(7573); break;
        case 961: SCRAMBLED_RINV(7577); break;
        case 962: SCRAMBLED_RINV(7583); break;
        case 963: SCRAMBLED_RINV(7589); break;
        case 964: SCRAMBLED_RINV(7591); break;
        case 965: SCRAMBLED_RINV(7603); break;
        case 966: SCRAMBLED_RINV(7607); break;
        case 967: SCRAMBLED_RINV(7621); break;
        case 968: SCRAMBLED_RINV(7639); break;
        case 969: SCRAMBLED_RINV(7643); break;
        case 970: SCRAMBLED_RINV(7649); break;
        case 971: SCRAMBLED_RINV(7669); break;
        case 972: SCRAMBLED_RINV(7673); break;
        case 973: SCRAMBLED_RINV(7681); break;
        case 974: SCRAMBLED_RINV(7687); break;
        case 975: SCRAMBLED_RINV(7691); break;
        case 976: SCRAMBLED_RINV(7699); break;
        case 977: SCRAMBLED_RINV(7703); break;
        case 978: SCRAMBLED_RINV(7717); break;
        case 979: SCRAMBLED_RINV(7723); break;
        case 980: SCRAMBLED_RINV(7727); break;
        case 981: SCRAMBLED_RINV(7741); break;
        case 982: SCRAMBLED_RINV(7753); break;
        case 983: SCRAMBLED_RINV(7757); break;
        case 984: SCRAMBLED_RINV(7759); break;
        case 985: SCRAMBLED_RINV(7789); break;
        case 986: SCRAMBLED_RINV(7793); break;
        case 987: SCRAMBLED_RINV(7817); break;
        case 988: SCRAMBLED_RINV(7823); break;
        case 989: SCRAMBLED_RINV(7829); break;
        case 990: SCRAMBLED_RINV(7841); break;
        case 991: SCRAMBLED_RINV(7853); break;
        case 992: SCRAMBLED_RINV(7867); break;
        case 993: SCRAMBLED_RINV(7873); break;
        case 994: SCRAMBLED_RINV(7877); break;
        case 995: SCRAMBLED_RINV(7879); break;
        case 996: SCRAMBLED_RINV(7883); break;
        case 997: SCRAMBLED_RINV(7901); break;
        case 998: SCRAMBLED_RINV(7907); break;
        case 999: SCRAMBLED_RINV(7919); break;
        case 1000: SCRAMBLED_RINV(7927); break;
        case 1001: SCRAMBLED_RINV(7933); break;
        case 1002: SCRAMBLED_RINV(7937); break;
        case 1003: SCRAMBLED_RINV(7949); break;
        case 1004: SCRAMBLED_RINV(7951); break;
        case 1005: SCRAMBLED_RINV(7963); break;
        case 1006: SCRAMBLED_RINV(7993); break;
        case 1007: SCRAMBLED_RINV(8009); break;
        case 1008: SCRAMBLED_RINV(8011); break;
        case 1009: SCRAMBLED_RINV(8017); break;
        case 1010: SCRAMBLED_RINV(8039); break;
        case 1011: SCRAMBLED_RINV(8053); break;
        case 1012: SCRAMBLED_RINV(8059); break;
        case 1013: SCRAMBLED_RINV(8069); break;
        case 1014: SCRAMBLED_RINV(8081); break;
        case 1015: SCRAMBLED_RINV(8087); break;
        case 1016: SCRAMBLED_RINV(8089); break;
        case 1017: SCRAMBLED_RINV(8093); break;
        case 1018: SCRAMBLED_RINV(8101); break;
        case 1019: SCRAMBLED_RINV(8111); break;
        case 1020: SCRAMBLED_RINV(8117); break;
        case 1021: SCRAMBLED_RINV(8123); break;
        case 1022: SCRAMBLED_RINV(8147); break;
        case 1023: SCRAMBLED_RINV(8161); break;
    }
    return std::min(inverse, ONE_MINUS_EPS);
}

MTS_NAMESPACE_END
