#include <mitsuba/core/plugin.h>
#include <mitsuba/core/bitmap.h>
#include <mitsuba/core/fstream.h>
#include <mitsuba/core/ray.h>
#include <mitsuba/core/fresolver.h>
#include <mitsuba/core/properties.h>
#include <mitsuba/render/util.h>
#include <mitsuba/render/scene.h>
#include <mitsuba/render/bsdf.h>
#include <mitsuba/render/medium.h>
#include <mitsuba/render/sampler.h>
#include <boost/filesystem/path.hpp>
#include "bsdfSimulator_proc.h"

MTS_NAMESPACE_BEGIN

class BSDFSimulator : public Utility {
public:
	int run(int argc, char **argv) {			
		m_wi = Vector(std::atof(argv[2]), std::atof(argv[3]), std::atof(argv[4]));
		m_sqrtNumParticles = std::atoi(argv[5]);
		m_size = std::atoi(argv[6]);	
		
		m_xmin = std::atof(argv[7]);
		m_xmax = std::atof(argv[8]);
		m_ymin = std::atof(argv[9]);
		m_ymax = std::atof(argv[10]);

		// Example: minDepth = 2, it will create 4 lobes:
		// 1st-order, 2nd-order, (3 to maxDepth)-th order, and all orders
		m_minDepth = std::atoi(argv[11]);
		m_maxDepth = std::atoi(argv[12]);
		m_shadowOption = std::atoi(argv[13]);

		if (argc > 15)
			m_useFullSphere = std::atoi(argv[15]);
		else
			m_useFullSphere = 0;

		if (argc > 16)
			m_distGiTexelScale = std::atof(argv[16]);

		ParameterMap params;
		if (argc > 17) {
			params["angular"] = argv[17];
			params["spatial"] = argv[18];
		}
		m_scene = loadScene(argv[1], params);
		
		// init
		m_scene->initialize();
		m_wi = normalize(m_wi);
		m_numParticles = m_sqrtNumParticles * m_sqrtNumParticles;

		Properties props = Properties("independent");
		props.setInteger("seed", 19931004);
		m_sampler = static_cast<Sampler *> (PluginManager::getInstance()->
			createObject(MTS_CLASS(Sampler), props));
		m_sampler->configure();

		ref<Scheduler> sched = Scheduler::getInstance();
		int sceneResID = sched->registerResource(m_scene);

		std::vector<SerializableObject *> samplers(sched->getCoreCount());
		for (size_t i = 0; i < sched->getCoreCount(); i++) {
			ref<Sampler> clonedSampler = m_sampler->clone();
			clonedSampler->incRef();
			samplers[i] = clonedSampler.get();
		}
		int samplerResID = sched->registerMultiResource(samplers);
		for (size_t i = 0; i < samplers.size(); i++)
			samplers[i]->decRef();

		ref<BSDFSimulatorProcess> proc = new BSDFSimulatorProcess(m_wi,
			m_sqrtNumParticles, m_size, AABB2(Point2(m_xmin, m_ymin), Point2(m_xmax, m_ymax)), 
			m_minDepth, m_maxDepth, m_shadowOption, m_useFullSphere, m_distGiTexelScale);
		proc->bindResource("scene", sceneResID);
		proc->bindResource("sampler", samplerResID);
		m_scene->bindUsedResources(proc);

		Log(EInfo, "Start rendering.");

		sched->schedule(proc);
		sched->wait(proc);

		Log(EInfo, "Finish rendering.");

		// Normalization
		double totValidParticles = (double)proc->m_res->getLobe(m_minDepth + 1)->m_totValidParticles;
		double totParticles = m_numParticles;
		double pScale = totParticles / totValidParticles;
		double totWeight = proc->m_res->getLobe(m_minDepth + 1)->m_totWeight;

		int numLobes = m_minDepth + 2;

		char txtFilename[256];
		FILE *fp;

		for (int i = 0; i < numLobes; i++) {
			char filename[256];
			if (m_minDepth == m_maxDepth && i + 1 != m_minDepth)
				continue;

			if (i < numLobes - 1)
				sprintf(filename, "%s_order_%d.exr", argv[14], i + 1);
			else
				sprintf(filename, "%s_order_all.exr", argv[14]);

			SphericalDistribution *lobe = proc->m_res->getLobe(i);
			
			Float invTotWeight = 0.0;
			if (totWeight > 1e-8)
				invTotWeight = 1.0 / totWeight;
			lobe->scale(totParticles * invTotWeight);
			lobe->saveExr(fs::path(filename));

			lobe->m_totValue *= invTotWeight;

			Vector3d &totalThroughput = lobe->m_totValue;
			Log(EInfo, "Total valid particles = %d / (%.0f, %d)", lobe->m_totValidParticles, 
				totValidParticles, m_numParticles);
			Log(EInfo, "Total thr = (%.6f, %.6f, %.6f)", totalThroughput[0], totalThroughput[1], totalThroughput[2]);
		}

		return 0;
	}

	ref<Scene> m_scene;
	ref<Sampler> m_sampler;
	Vector m_wi;
	int m_numParticles, m_sqrtNumParticles;
	int m_size;
	int m_minDepth;
	int m_maxDepth;
	double m_xmin, m_xmax, m_ymin, m_ymax;
	int m_shadowOption;

	int m_useFullSphere;
	Float m_distGiTexelScale;

	MTS_DECLARE_UTILITY()
};

MTS_EXPORT_UTILITY(BSDFSimulator, "BSDF simulator")
MTS_NAMESPACE_END
