#pragma once

#include <core/ptr.h>
#include <render/medium.h>
#include <render/volumegrid.h>

#include <sstream>
#include <string>

struct Homogeneous : Medium {
    Medium *clone() const override {
        return new Homogeneous(*this);
    }

    void merge(Medium *other) override {
        Homogeneous *m = dynamic_cast<Homogeneous *>(other);
        sigma_t += m->sigma_t;
        albedo.merge(m->albedo);
    }

    Homogeneous(const Properties &props) : Medium(props) {
        this->m_type = TYPE_ID;
        sigma_t = props.get<Float>("sigmaT");
        if (props.has<Spectrum>("albedo"))
            albedo = VolumeGrid(props.get<Spectrum>("albedo"));
        else if (props.has<Properties>("albedo"))
            albedo = VolumeGrid(props.get<Properties>("albedo"));
        else
            Throw("No albedo specified for homogeneous medium!");
        sampling_weight = props.get<Float>("sampling_weight", 1.);
    }

    inline Homogeneous(Float sigma_t, const Spectrum &albedo, int phase_id)
        : Medium(phase_id), sigma_t(sigma_t), albedo(albedo) {
        this->m_type = TYPE_ID;
        sampling_weight = 1.; // FIXME
        // sampling_weight = albedo.maxCoeff();
    }

    inline Homogeneous(Float sigma_t, const VolumeGrid &albedo, int phase_id)
        : Medium(phase_id), sigma_t(sigma_t), albedo(albedo) {
        this->m_type = TYPE_ID;
        sampling_weight = 1.; // FIXME
        // sampling_weight = albedo.maxCoeff();
    }

    void setZero() override {
        sigma_t = 0.f;
        albedo.setZero();
        sampling_weight = 0.f;
    }

    bool sampleDistance(const Ray &ray, const Float &tmax,
                        const Array2 &rnd2, RndSampler *sampler,
                        Vector &p_scatter, Spectrum &throughput) const override;

    bool sampleDistance(const Ray &ray, const Float &tmax,
                        RndSampler           *sampler,
                        MediumSamplingRecord &mRec) const override;

    Float evalTransmittance(const Ray &ray, const Float &tmin,
                            const Float &tmax, RndSampler *sampler) const;

    Float evalTransmittanceRatio(const Ray &ray, const Float &tmin,
                                 const Float &tmax, RndSampler *sampler) const;

    inline bool isHomogeneous() const override { return true; }

    inline Spectrum    sigS(const Vector &x) const { return sigma_t * this->albedo.lookupSpectrum(x); }
    inline Float       sigT(const Vector &x) const { return sigma_t; };
    inline std::string toString() const {
        std::ostringstream oss;
        oss << "Homogeneous Medium...";
        return oss.str();
    }

    Float      sigma_t;
    VolumeGrid albedo;
    Float      sampling_weight;

    static const int TYPE_ID = 1;

    PSDR_DECLARE_CLASS(Homogeneous)
    PSDR_IMPLEMENT_VIRTUAL_CLASS(Homogeneous)
};

PSDR_DECLARE_MEDIUM_HELPER_FUNCTIONS(Homogeneous);
