#pragma once

#include <core/fwd.h>
#include <core/object.h>

struct Intersection;

enum EBSDFMode {
    ERadiance                 = 0,
    EImportance               = 1,
    EImportanceWithCorrection = 2
};

enum EBSDF {
    EDiffuseBSDF = 0,
};

using BSDFEvalType = Spectrum;

struct BSDF : public Object {
    virtual ~BSDF() = default;

    virtual BSDF *clone() const     = 0;
    virtual void  merge(BSDF *bsdf) = 0;
    virtual void  setZero()         = 0;

    // Evaluate the cosine weighted BSDF value
    BSDFEvalType eval(const Intersection &its, const Vector &wo,
                      EBSDFMode mode = EBSDFMode::ERadiance) const;
    Float        correction(const Intersection &its, const Vector &wo) const;

    // Sample the BSDF and return the BSDF value divided by pdf
    BSDFEvalType sample(const Intersection &its, const Array3 &sample,
                        Vector &wo, Float &pdf, Float &eta,
                        EBSDFMode mode = EBSDFMode::ERadiance) const;

    // Compute the probability of sampling wo (given wi).
    //! This function should be marked as inactive
    Float pdf(const Intersection &its, const Vector3 &wo) const;

    // Check if the BSDF is transmissive
    bool isTransmissive() const;

    // Check if the BSDF is two-sided
    bool isTwosided() const;

    // Check if the BSDF is null
    bool isNull() const;

    /// Return a readable string representation of this BSDF
    virtual std::string toString() const;

    virtual std::unordered_map<std::string, ParamType> toMap() const {
        return {};
    };

    PSDR_DECLARE_VIRTUAL_CLASS()

    // --------------------------------------------------
    int m_type = -1;
};

#define PSDR_DECLARE_BSDF_HELPER_EVAL(CLASSNAME)                              \
    void __##CLASSNAME##_eval(const CLASSNAME *bsdf, const Intersection &its, \
                              const Vector &wo, EBSDFMode mode,               \
                              BSDFEvalType &res)

#define PSDR_DECLARE_BSDF_HELPER_SAMPLE(CLASSNAME) \
    void __##CLASSNAME##_sample(const CLASSNAME *bsdf, const Intersection &its, const Array3 &sample, Vector &wo, Float &pdf, Float &eta, EBSDFMode mode, BSDFEvalType &res)

#define PSDR_DECLARE_BSDF_HELPER_PDF(CLASSNAME) \
    void __##CLASSNAME##_pdf(const CLASSNAME *bsdf, const Intersection &its, const Vector3 &wo, Float &res)

#define PSDR_DECLARE_BSDF_HELPER_ISNULL(CLASSNAME) \
    void __##CLASSNAME##_isNull(const CLASSNAME *bsdf, bool &res)

#define PSDR_INVOKE_BSDF_HELPER_EVAL(CLASSNAME, bsdf, its, wo, mode, res) \
    __##CLASSNAME##_eval(dynamic_cast<const CLASSNAME *>(bsdf), its, wo, mode, res)

#define PSDR_INVOKE_BSDF_HELPER_SAMPLE(CLASSNAME, bsdf, its, sample, wo, pdf, eta, mode, res) \
    __##CLASSNAME##_sample(dynamic_cast<const CLASSNAME *>(bsdf), its, sample, wo, pdf, eta, mode, res)

#define PSDR_INVOKE_BSDF_HELPER_PDF(CLASSNAME, bsdf, its, wo, res) \
    __##CLASSNAME##_pdf(dynamic_cast<const CLASSNAME *>(bsdf), its, wo, res)

#define PSDR_INVOKE_BSDF_HELPER_ISNULL(CLASSNAME, bsdf, res) \
    __##CLASSNAME##_isNull(dynamic_cast<const CLASSNAME *>(bsdf), res)

#define PSDR_DECLARE_BSDF_HELPER_FUNCTIONS(CLASSNAME) \
    PSDR_DECLARE_BSDF_HELPER_EVAL(CLASSNAME);         \
    PSDR_DECLARE_BSDF_HELPER_SAMPLE(CLASSNAME);       \
    PSDR_DECLARE_BSDF_HELPER_PDF(CLASSNAME);          \
    PSDR_DECLARE_BSDF_HELPER_ISNULL(CLASSNAME)

#define PSDR_IMPL_BSDF_HELPER_EVAL(CLASSNAME)                                 \
    void __##CLASSNAME##_eval(const CLASSNAME *bsdf, const Intersection &its, \
                              const Vector &wo, EBSDFMode mode,               \
                              BSDFEvalType &res) {                            \
        res = bsdf->eval(its, wo, mode);                                      \
    }

#define PSDR_IMPL_BSDF_HELPER_SAMPLE(CLASSNAME)                                                                                                                                \
    void __##CLASSNAME##_sample(const CLASSNAME *bsdf, const Intersection &its, const Array3 &sample, Vector &wo, Float &pdf, Float &eta, EBSDFMode mode, BSDFEvalType &res) { \
        res = bsdf->sample(its, sample, wo, pdf, eta, mode);                                                                                                                   \
    }

#define PSDR_IMPL_BSDF_HELPER_PDF(CLASSNAME)                                                                  \
    void __##CLASSNAME##_pdf(const CLASSNAME *bsdf, const Intersection &its, const Vector3 &wo, Float &res) { \
        res = bsdf->pdf(its, wo);                                                                             \
    }

#define PSDR_IMPL_BSDF_HELPER_ISNULL(CLASSNAME)                     \
    void __##CLASSNAME##_isNull(const CLASSNAME *bsdf, bool &res) { \
        res = bsdf->isNull();                                       \
    }

#define PSDR_IMPL_BSDF_HELPER_FUNCTIONS(CLASSNAME) \
    PSDR_IMPL_BSDF_HELPER_EVAL(CLASSNAME);         \
    PSDR_IMPL_BSDF_HELPER_SAMPLE(CLASSNAME);       \
    PSDR_IMPL_BSDF_HELPER_PDF(CLASSNAME);          \
    PSDR_IMPL_BSDF_HELPER_ISNULL(CLASSNAME)