from typing import List
import gin
from pypsdr.plot import write_obj
import psdr_cpu
import dataclasses
from dataclasses import field
from pypsdr.validate import *
from pypsdr.utils.io import *
from pypsdr.loss import compute_render_loss, uniform_laplacian, L2_loss, laplace_regularizer_const
from pypsdr.optimize import Camera
from pypsdr.optimizer import LGDescent,sparse_eye, LargeSteps, LargeSteps2, LargeStepsOptimizer
from pypsdr.renderer import Render
from pypsdr.common import gen_cameras
os.environ["OPENCV_IO_ENABLE_OPENEXR"]="1"

from random import randrange, seed

import scipy
import scipy.sparse.linalg
import datetime


@gin.configurable
# generate the constructor
@dataclasses.dataclass
class TrainRunner:
    scene_init_file: str = "./kitty/ini.xml"
    scene_target_file: str = "./kitty/tar.xml"
    shape_id: int = 0  # shape to be optimized
    niter: int = 500
    options: psdr_cpu.RenderOptions = None
    guiding_options: GuidingOptions = None
    sample_edge: bool = False
    out_dir: str = "./output"          # output directory
    # directory that contains target images
    target_dir: str = "./kitty/output/kitty/target"
    test_indices: List = field(
        default_factory=lambda: [0, 14, 32, 57])

    lr: float = 1.0
    lmbda: float = 500.0
    batch_size: int = 10
    print_size: int = 10



    camera_file: str = "./kitty/cam_pos.txt"
    integrator: psdr_cpu.Integrator = psdr_cpu.Path2()
    mala_options: psdr_cpu.MALAOptions = psdr_cpu.MALAOptions()

    def __post_init__(self):
        mkdir(self.out_dir)
        mkdir(os.path.join(self.out_dir, "iter"))
        self.scene_init = Scene(self.scene_init_file)
        self.width = self.scene_init.camera.width
        self.height = self.scene_init.camera.height
        self.fov = self.scene_init.camera.fov
        self.train_images = np.array(read_images(self.target_dir))
        self.test_images = self.train_images[self.test_indices]
        self.test_image = np.concatenate(self.test_images, axis=1)
        print(self.test_image.shape)
        imwrite(self.test_image,
                os.path.join(self.out_dir, "test.png"))

        self.train_images = torch.from_numpy(self.train_images)
        self.test_images = torch.from_numpy(self.test_images)

        camera_pos = np.loadtxt(self.camera_file)
        camera_info = Camera(target=gin.REQUIRED)

        self.cameras = gen_cameras(positions=camera_pos,
                                   target=camera_info.target,
                                   up = camera_info.up,
                                   fov=camera_info.fov,
                                   resolution=[self.width, self.height], type=0)
        # self.cameras = [self.scene_init.camera]
        self.sceneAD = psdr_cpu.SceneAD(self.scene_init)
        self.render = Render(self.sceneAD, self.cameras,
                             self.integrator, self.options, self.shape_id, self.guiding_options, self.mala_options, self.sample_edge)
        psdr_cpu.set_verbose(True)

    def run(self):
        guiding_options = GuidingOptions()

        F = torch.tensor(np.array(self.scene_init.shapes[self.shape_id].indices)).long()
        E = torch.tensor(
            np.array(self.scene_init.shapes[self.shape_id].getEdges())).long()
        V = torch.tensor(
            np.array(self.scene_init.shapes[self.shape_id].vertices),
            dtype=torch.float32,
            requires_grad=True)
        
        # print(F.shape, E.shape, V.shape)
        # exit()
        
        lap_regularizer_fn = laplace_regularizer_const(V, E, F)

        print('Starting optimization')
        # L = uniform_laplacian(V, E).detach() * \
        #     self.lmbda
        # I = sparse_eye(L.shape[0])
        # IL_term = I + L
        # print('IL_term constructed')
        # Lv = np.asarray(IL_term.coalesce().values())
        # Li = np.asarray(IL_term.coalesce().indices())
        # IL_term_s = scipy.sparse.coo_matrix((Lv, Li), shape=L.shape)
        # print('IL_term_s constructed')
        # IL_term_s_solver = scipy.sparse.linalg.factorized(IL_term_s)
        
        # # print(np.max(np.abs(IL_term_s))) 
        # # exit()
        # print('IL_term_s_solver constructed')
        # optimizer = LGDescent(
        #     params=[
        #         {'params': V,
        #          'lr':  self.lr},
        #     ],
        #     IL_term=IL_term,
        #     IL_solver=IL_term_s_solver)
        optimizer = LargeStepsOptimizer(V, F, self.lr, lmbda=self.lmbda)
        
        # optimizer = torch.optim.Adam([V], lr=self.lr)
        

        ### WARN: add edge sort here ###
        if guiding_options.guide_type != "":
            self.sceneAD.val.shapes[self.shape_id].sort_config = guiding_options.sort_config
            self.sceneAD.val.shapes[self.shape_id].enable_draw = True
        
        for i in range(len(self.scene_init.shapes)):
            self.sceneAD.val.shapes[i].enable_edge = False
        self.sceneAD.val.shapes[self.shape_id].enable_edge = True
        self.sceneAD.val.shapes[self.shape_id].configure()

        seed(1)
        error = []

        for iter in range(self.niter):
            now = datetime.datetime.now()

            options = RenderOptions(
                iter, gin.REQUIRED, gin.REQUIRED, gin.REQUIRED)
            mala_options = MALAOptions(gin.REQUIRED)
            optimizer.zero_grad()
            loss = 0.0


            for j in range(self.batch_size):
                sensor_id = randrange(len(self.train_images))
                print("Rendering camera:", sensor_id, end='\r')
                img = self.render(V, sensor_id, {"options": options, "guiding_options": guiding_options, "mala_options": mala_options})
                # img[img.isnan()] = 0.0
                img_loss = compute_render_loss(
                    img, self.train_images[sensor_id], 1.0)
                # geo_loss = lap_regularizer_fn.eval(V) * self.lmbda * 10000
                ttl_loss = img_loss
                loss += img_loss.item()
                ttl_loss.backward()
                print("Rendering camera:", sensor_id,
                      "loss:", img_loss.item(), end='\r')
            V.grad[0] += 0.0000000001
            optimizer.step()
            # update scene
            self.sceneAD.val.shapes[self.shape_id].setVertices(
                V.detach().numpy())
            self.render.setState({'sceneAD': self.sceneAD})

            # print stat
            print("grad: ", np.abs(V.grad).sum().item())
            end = datetime.datetime.now() - now
            print('Iteration:', iter,
                  'Loss:', loss, 
                  'Total time:', end.seconds + end.microseconds / 1000000.0)
            error.append(loss)
            np.savetxt(self.out_dir+"/loss.log", error)

            # write files
            if iter % self.print_size == 0:
                write_obj(V.detach(), F.detach(),
                          os.path.join(self.out_dir, "iter", "iter_%d.obj" % iter))
                test_images = []
                for test_id in self.test_indices:
                    print("Testing camera:", test_id, end='\r')
                    test_img = self.render(V.detach(), test_id,
                                           {"options": RenderOptions(iter, gin.REQUIRED, gin.REQUIRED, 0, 0)})
                    test_images.append(test_img)
                test_image = np.concatenate(test_images, axis=1)
                imwrite(np.concatenate([self.test_image, test_image], axis=0),
                        os.path.join(self.out_dir, "iter", "iter_%d.exr" % iter))


import argparse
if __name__ == "__main__":
    # os.chdir('./inverse_rendering')
    default_config = './cbox_bunny_mirror.conf'
    parser = argparse.ArgumentParser(
        description='Script for generating validation results')
    parser.add_argument('config_file', metavar='config_file',
                        type=str, nargs='?', default=default_config, help='config file')
    args, unknown = parser.parse_known_args()
    # Dependency injection: Arguments are injected into the function from the gin config file.
    gin.parse_config_file(args.config_file, skip_unknown=True)
    train_runner = TrainRunner()
    train_runner.run()