from typing import List
import gin
from pypsdr.plot import write_obj
import psdr_cpu
import dataclasses
from dataclasses import field
from pypsdr.validate import *
from pypsdr.utils.io import *
from pypsdr.loss import compute_render_loss, uniform_laplacian, L2_loss
from pypsdr.optimize import Camera
from pypsdr.optimizer import LGDescent,sparse_eye, LargeStepsOptimizer
from pypsdr.renderer import Render2, RenderMi
from scipy.spatial.transform import Rotation as R
os.environ["OPENCV_IO_ENABLE_OPENEXR"]="1"

from random import randrange, seed

import scipy
import scipy.sparse.linalg
import datetime

import drjit as dr
import mitsuba as mi
import matplotlib.pyplot as plt # We'll also want to plot some outputs
import os

mi.set_variant('llvm_ad_rgb')

@gin.configurable
# generate the constructor
@dataclasses.dataclass
class TrainRunner:
    scene_init_file: str = "./kitty/ini.xml"
    scene_init_file_mi: str = "./kitty/ini.xml"
    scene_target_file: str = "./kitty/tar.xml"
    shape_id: int = 0  # shape to be optimized
    niter: int = 500
    options: psdr_cpu.RenderOptions = None
    guiding_options: GuidingOptions = None
    out_dir: str = "./output"          # output directory
    # directory that contains target images
    target_dir: str = "./kitty/output/kitty/target"
    test_indices: List = field(
        default_factory=lambda: [0, 14, 32, 57])

    lr: float = 1.0
    lmbda: float = 500.0
    batch_size: int = 10
    print_size: int = 10

    integrator: psdr_cpu.Integrator = psdr_cpu.Path2()
    mala_options: psdr_cpu.MALAOptions = psdr_cpu.MALAOptions()

    def __post_init__(self):
        mkdir(self.out_dir)
        mkdir(os.path.join(self.out_dir, "iter"))
        self.train_images = np.array(read_images(self.target_dir))
        self.test_images = self.train_images[self.test_indices]
        self.test_image = np.concatenate(self.test_images, axis=1)
        imwrite(self.test_image,
                os.path.join(self.out_dir, "test.png"))

        self.train_images = torch.from_numpy(self.train_images)
        self.test_images = torch.from_numpy(self.test_images)
        self.key = 'mesh.vertex_positions'
        self.key_f = 'mesh.faces'
        self.scene_mi = mi.load_file(self.scene_init_file_mi)
        integrator_mi = mi.load_dict({
            'type': 'prb_projective',
            'sppc': self.options.spp,
            'sppp': self.options.sppe,
            'sppi': 0,
            'max_depth': self.options.max_bounces + 1,
        })
        self.integrator_mi = integrator_mi
        self.scene_init = Scene(self.scene_init_file)
        self.width = self.scene_init.camera.width
        self.height = self.scene_init.camera.height
        self.obj_pos = np.loadtxt('obj_pos.txt')
            
        for i in range(len(self.scene_init.shapes)):
            self.scene_init.shapes[i].enable_edge = False
        self.scene_init.shapes[self.shape_id].enable_edge = True
        self.scene_init.configure()
        self.render = RenderMi(self.integrator_mi, self.obj_pos)
        print("config fin")
        
        psdr_cpu.set_verbose(True)


    def run(self):
        params = mi.traverse(self.scene_mi)
        
        F = torch.tensor(np.array(self.scene_init.shapes[self.shape_id].indices), dtype=torch.int32)
        E = torch.tensor(
            self.scene_init.shapes[self.shape_id].getEdges()).long()
        V = torch.tensor(
            np.array(self.scene_init.shapes[self.shape_id].vertices_world),
            dtype=torch.float32,
            requires_grad=True)
        
        
        F_mi = dr.llvm.ad.Int(F.flatten())
        params[self.key_f] = F_mi
        
        params.update()

        print('Starting optimization')
        # L = uniform_laplacian(V, E).detach() * \
        #     self.lmbda
        # I = sparse_eye(L.shape[0])
        # IL_term = I + L
        # Lv = np.asarray(IL_term.coalesce().values())
        # Li = np.asarray(IL_term.coalesce().indices())
        # IL_term_s = scipy.sparse.coo_matrix((Lv, Li), shape=L.shape)
        # IL_term_s_solver = scipy.sparse.linalg.factorized(IL_term_s)

        # optimizer = LGDescent(
        #     params=[
        #         {'params': V,
        #          'lr':  self.lr},
        #     ],
        #     IL_term=IL_term,
        #     IL_solver=IL_term_s_solver)
        optimizer = LargeStepsOptimizer(V, F, self.lr, lmbda=self.lmbda)
        
        seed(1)
        error = []
        params = mi.traverse(self.scene_mi)

        for iter in range(self.niter):
            now = datetime.datetime.now()
            optimizer.zero_grad()

            print(self.batch_size)

            for j in range(self.batch_size):
                self.options.seed = iter
                pos_id = (iter*self.batch_size+j) % len(self.train_images)
                raw_position = V.detach().numpy()
                pos = self.obj_pos[pos_id]
                r = R.from_rotvec(pos[3] * pos[0:3])
                rot_position = np.dot(raw_position, r.as_matrix().transpose())
                self.scene_init.shapes[0].setVertices(rot_position)

                self.scene_init.shapes[0].configure()  # !
                self.scene_init.configure()
                # img_fwd = self.integrator.renderC(self.scene_init, self.options)\
                #     .reshape(self.height, self.width, 3)
                # img_fwd = torch.tensor(img_fwd, dtype=torch.float32)
                img = self.render(self.scene_mi, params, self.key, iter + 600, self.options.spp, V, pos_id)
                
                # img = render_mesh(self.scene_mi, V.flatten(), spp = self.options.spp, seed=iter)
                # print(img)
                # L1 Loss
                loss = torch.sum(torch.abs(img - self.train_images[pos_id]))
                
                loss.backward()
                print("Rendering camera:", pos_id,
                        "loss:", loss)
                # if j == 0:
                #     Vg = V.grad.clone()
                # else:
                #     Vg += V.grad.clone()
                # del img_loss
            # V.grad = Vg
            # update scene
            optimizer.step()


            # raw_position = V.detach().numpy()
            # pos = self.obj_pos[pos_id]
            # r = R.from_rotvec(pos[3] * pos[0:3])
            # rot_position = np.dot(raw_position, r.as_matrix().transpose())
            # self.sceneAD.val.shapes[self.shape_id].setVertices(rot_position)

            # print stat
            end = datetime.datetime.now() - now
            print("grad: ", np.abs(V.grad).sum().item())
            print('Iteration:', iter,
                  'Loss:', loss.detach().numpy(),
                  'Total time:', end.seconds + end.microseconds / 1000000.0)
            error.append(loss.detach().numpy())
            np.savetxt(self.out_dir+"/loss.log", error)

            # write files
            if iter % self.print_size == 0:
                write_obj(V.detach(), F.detach(),
                          os.path.join(self.out_dir, "iter", "iter_%d.obj" % iter))
                test_images = []
                for test_id in self.test_indices:
                    print("Testing camera:", test_id, end='\r')
                    raw_position = V.detach().numpy()
                    pos = self.obj_pos[test_id]
                    r = R.from_rotvec(pos[3] * pos[0:3])
                    rot_position = np.dot(raw_position, r.as_matrix().transpose())
                    self.scene_init.shapes[0].setVertices(rot_position)

                    self.scene_init.shapes[0].configure()  # !
                    self.scene_init.configure()
                    # test_img = self.integrator.renderC(self.scene_init, self.options)\
                    #     .reshape(self.height, self.width, 3)
                    test_img = self.render(self.scene_mi, params, self.key, iter + 400, self.options.spp, V, test_id, None).detach().numpy()
                    test_images.append(test_img)
                test_image = np.concatenate(test_images, axis=1)
                imwrite(np.concatenate([self.test_image, test_image], axis=0),
                        os.path.join(self.out_dir, "iter", "iter_%d.exr" % iter))
            # exit()




import argparse
if __name__ == "__main__":
    default_config = './mirror_multiview1_mi.conf'
    parser = argparse.ArgumentParser(
        description='Script for generating validation results')
    parser.add_argument('config_file', metavar='config_file',
                        type=str, nargs='?', default=default_config, help='config file')
    args, unknown = parser.parse_known_args()
    # Dependency injection: Arguments are injected into the function from the gin config file.
    gin.parse_config_file(args.config_file, skip_unknown=True)

    train_runner = TrainRunner()
    train_runner.run()