from typing import List
import gin
from pypsdr.plot import write_obj
import psdr_cpu
import dataclasses
from dataclasses import field
from pypsdr.validate import *
from pypsdr.utils.io import *
from pypsdr.loss import compute_render_loss, uniform_laplacian
from pypsdr.optimize import Camera
from pypsdr.optimizer import LGDescent,sparse_eye
from pypsdr.renderer import Render
from pypsdr.common import gen_cameras

from random import randrange, seed

import scipy
import scipy.sparse.linalg
import datetime


@gin.configurable
# generate the constructor
@dataclasses.dataclass
class TrainRunner:
    scene_init_file: str = "./kitty/ini.xml"
    scene_target_file: str = "./kitty/tar.xml"
    shape_id: int = 0  # shape to be optimized
    niter: int = 500
    options: psdr_cpu.RenderOptions = None
    out_dir: str = "./output"          # output directory
    # directory that contains target images
    target_dir: str = "./kitty/output/kitty/target"
    test_indices: List = field(
        default_factory=lambda: [0, 14, 32, 57])
    img_threshold: float = 2.
    lr: float = 1.0
    lmbda: float = 500.0
    batch_size: int = 10
    print_size: int = 10



    camera_file: str = "./kitty/cam_pos.txt"
    integrator: psdr_cpu.Integrator = psdr_cpu.Direct()

    def __post_init__(self):
        mkdir(self.out_dir)
        mkdir(os.path.join(self.out_dir, "iter"))
        self.scene_init = Scene(self.scene_init_file)
        self.width = self.scene_init.camera.width
        self.height = self.scene_init.camera.height
        self.fov = self.scene_init.camera.fov
        self.train_images = np.array(read_images(self.target_dir))
        self.test_images = self.train_images[self.test_indices]
        self.test_image = np.concatenate(self.test_images, axis=1)
        imwrite(self.test_image,
                os.path.join(self.out_dir, "test.png"))

        self.train_images = torch.from_numpy(self.train_images)
        self.test_images = torch.from_numpy(self.test_images)

        camera_pos = np.loadtxt(self.camera_file)
        camera_info = Camera(target=gin.REQUIRED)

        self.cameras = gen_cameras(positions=camera_pos,
                                   target=camera_info.target,
                                   up = camera_info.up,
                                   fov=camera_info.fov,
                                   resolution=[self.width, self.height])
        self.sceneAD = psdr_cpu.SceneAD(self.scene_init)
        self.render = Render(self.sceneAD, self.cameras,
                             self.integrator, self.options, self.shape_id)
        # psdr_cpu.set_verbose(True)

    def run(self):
        F = torch.tensor(self.scene_init.shapes[self.shape_id].indices).long()
        E = torch.tensor(
            self.scene_init.shapes[self.shape_id].getEdges()).long()
        V = torch.tensor(
            self.scene_init.shapes[self.shape_id].vertices,
            dtype=torch.float32,
            requires_grad=True)

        print('Starting optimization')
        L = uniform_laplacian(V, E).detach() * \
            self.lmbda
        I = sparse_eye(L.shape[0])
        IL_term = I + L
        Lv = np.asarray(IL_term.coalesce().values())
        Li = np.asarray(IL_term.coalesce().indices())
        IL_term_s = scipy.sparse.coo_matrix((Lv, Li), shape=L.shape)
        IL_term_s_solver = scipy.sparse.linalg.factorized(IL_term_s)

        optimizer = LGDescent(
            params=[
                {'params': V,
                 'lr':  self.lr},
            ],
            IL_term=IL_term,
            IL_solver=IL_term_s_solver)
        seed(1)
        np.random.seed(0)
        idx_shuffled = np.arange(len(self.train_images))
        np.random.shuffle(idx_shuffled)
        error = []
        for iter in range(self.niter):
            now = datetime.datetime.now()
            options = RenderOptions(
                iter, gin.REQUIRED, gin.REQUIRED, gin.REQUIRED)
            optimizer.zero_grad()
            loss = 0.0
            for j in range(self.batch_size):
                sensor_id = idx_shuffled[j]
                img = self.render(V, sensor_id, {"options": options})
                tar = self.train_images[sensor_id]
                #clamping
                img = img.clamp(0., self.img_threshold)
                tar = tar.clamp(0., self.img_threshold)
                img_loss = compute_render_loss(
                    img, tar, 1.0)
                loss = img_loss.item()
                img_loss.backward()
                print("Rendering camera:", sensor_id,
                      "loss:", img_loss.item(), end='\r')
            if V.grad.sum().abs() > 0:
                optimizer.step()

            # update scene
            self.sceneAD.val.shapes[self.shape_id].setVertices(
                V.detach().numpy())
            self.render.setState({'sceneAD': self.sceneAD})

            # print stat
            print("grad: ", np.abs(V.grad).sum().item())
            end = datetime.datetime.now() - now
            print('Iteration:', iter,
                  'Loss:', loss,
                  'Total time:', end.seconds + end.microseconds / 1000000.0)
            error.append(loss)
            np.savetxt(os.path.join(self.out_dir, "loss.log"), error)
            
            # write files
            if iter % self.print_size == 0:
                write_obj(V.detach(), F.detach(),
                          os.path.join(self.out_dir, "iter", "iter_%d.obj" % iter))
                test_images = []
                for test_id in self.test_indices:
                    print("Testing camera:", test_id, end='\r')
                    test_img = self.render(V.detach(), test_id,
                                           {"options": RenderOptions(iter, gin.REQUIRED, 128, 0, 0)})
                    test_images.append(test_img)
                test_image = np.concatenate(test_images, axis=1)
                imwrite(np.concatenate([self.test_image, test_image], axis=0),
                        os.path.join(self.out_dir, "iter", "iter_%d.png" % iter))


import argparse
if __name__ == "__main__":
    default_config = './kitty.conf'
    parser = argparse.ArgumentParser(
        description='Script for generating validation results')
    parser.add_argument('config_file', metavar='config_file',
                        type=str, nargs='?', default=default_config, help='config file')
    args, unknown = parser.parse_known_args()
    # Dependency injection: Arguments are injected into the function from the gin config file.
    gin.parse_config_file(args.config_file, skip_unknown=True)
    train_runner = TrainRunner()
    train_runner.run()