#pragma once
#include <bsdf/microfacet.h>
#include <core/ptr.h>
#include <render/bsdf.h>

struct RoughDielectricBSDF : BSDF {
    RoughDielectricBSDF(Float alpha, Float intIOR, Float extIOR) : m_distr(alpha) {
        m_eta    = intIOR / extIOR;
        m_invEta = 1.0f / m_eta;
        this->m_type = this->TYPE_ID;
    }

    RoughDielectricBSDF(Float alpha, Float intIOR, Float extIOR,
                               Spectrum specularReflectance, Spectrum specularTransmittance)
        : m_distr(alpha),
          m_specularReflectance(specularReflectance),
          m_specularTransmittance(specularTransmittance) {
        m_eta    = intIOR / extIOR;
        m_invEta = 1.0f / m_eta;
        this->m_type = this->TYPE_ID;
    }

    BSDF *clone() const override;
    void merge(BSDF *bsdf) override {
        RoughDielectricBSDF *_bsdf = dynamic_cast<RoughDielectricBSDF *>(bsdf);
        m_eta += _bsdf->m_eta;
        m_distr.m_alpha += _bsdf->m_distr.m_alpha;
    }

    void setZero() override {
        m_distr.setZero();
        m_eta = 0;
    }

    Spectrum eval(const Intersection &its, const Vector &wo,
                  EBSDFMode mode = EBSDFMode::ERadiance) const;

    Spectrum sample(const Intersection &its, const Array3 &rnd, Vector &wo,
                    Float &pdf, Float &eta,
                    EBSDFMode mode = EBSDFMode::ERadiance) const;
    Float pdf(const Intersection &its, const Vector &wo) const;

    inline bool isTransmissive() const { return true; }
    inline bool isTwosided() const { return true; }
    inline bool isNull() const { return false; }

    std::string toString() const override {
        std::ostringstream oss;
        oss << "BSDF_rough_dielectric [" << '\n'
            //<< "  alpha = " << m_alpha << '\n'
            << "  eta = " << m_eta << '\n'
            << "]" << std::endl;
        return oss.str();
    }

    MicrofacetDistribution m_distr;
    Float m_eta, m_invEta;
    Spectrum m_specularReflectance = Spectrum::Ones();
    Spectrum m_specularTransmittance = Spectrum::Ones();

    static const int TYPE_ID = 3;

    PSDR_DECLARE_CLASS(RoughDielectricBSDF)
    PSDR_IMPLEMENT_VIRTUAL_CLASS(RoughDielectricBSDF)
};

PSDR_DECLARE_BSDF_HELPER_FUNCTIONS(RoughDielectricBSDF);