#include "envmap.h"

#include <core/transform.h>
#include <core/bitmap.h>
#include <math.h>
#include <render/intersection.h>

EnvironmentMap::EnvironmentMap(const Properties &props) : Emitter(props) {
    if (props.has("filename")) {
        m_data.load(props.get<std::string>("filename").c_str());
    }
    else if (props.has("data")) {
        m_data = props.get<Bitmap>("data");
    }
    else {
        throw std::runtime_error("No filename or data for EnvironmentMap!");
    }
    m_to_world = props.get<Matrix4x4>("toWorld", Matrix4x4::Identity());
    m_scale    = props.get<Float>("scale", 1.0f);
    this->m_type = this->TYPE_ID;
    configure();
}

EnvironmentMap::EnvironmentMap(const std::string filename, int shape_id)
    : Emitter(shape_id) {
    m_data.load(filename.c_str());
    m_to_world = Matrix4x4::Identity();
    m_scale    = 1.0f;
    this->m_type = this->TYPE_ID;
    configure();
}

void EnvironmentMap::configure() {
    int width  = m_data.width();
    int height = m_data.height();
    PSDR_ASSERT(width > 1 && height > 1);
    width  = (width - 1) << 1;
    height = (height - 1) << 1;
    m_cube_distrb.set_resolution(Array2i(width, height));
    m_cube_distrb.set_mass(m_data);

    m_to_local = m_to_world.inverse();
    m_ready    = true;
}

Emitter *EnvironmentMap::clone() const {
    return new EnvironmentMap(*this);
}

void EnvironmentMap::merge(Emitter *emitter) {
    m_data += dynamic_cast<EnvironmentMap *>(emitter)->m_data;
};

void EnvironmentMap::setZero() {
    m_data.setZero();
}

Spectrum EnvironmentMap::eval(const Intersection &, const Vector &d) const {
    // Vector3 wi_world = its.toWorld(d);
    // NOTE only the direction matters
    return eval(Vector(0, 0, 0) /*unused*/, d); // NOTE -wi_world
}

Spectrum EnvironmentMap::eval(const Vector &, const Vector &d) const {
    PSDR_ASSERT(m_ready);
    Vector3 v = psdr::transform_dir(m_to_local, -d);
    Vector2 uv(atan2(v.x(), -v.z()) * INV_TWOPI, acos(v.y()) * INV_PI);
    // Note: Enzyme issue
    // uv -= uv.array().floor().matrix();
    uv.x() -= std::floor(uv.x());
    uv.y() -= std::floor(uv.y());
    return m_data.eval(uv, false /* flip_v */) * m_scale;
}

Float EnvironmentMap::evalDirection(const Vector &, const Vector &) const {
    assert(false);
    return 0.;
}

Float EnvironmentMap::sampleDirection(const Array2 &, Vector &, Float *) const {
    assert(false);
    return 0.;
}

Float EnvironmentMap::pdf(const Intersection &, const Vector &) const {
    assert(false);
    return 0.;
}

Float EnvironmentMap::pdfDirect(const DirectSamplingRecord &dRec) const {
    Vector dir  = dRec.p - dRec.ref;
    Float  dist = dir.norm();
    dir /= dist;
    Float G = std::abs(dRec.n.dot(dir)) / (dist * dist);
    dir     = psdr::transform_dir(m_to_local, dir);
    Vector2 uv(atan2(dir.x(), -dir.z()) * INV_TWOPI, acos(dir.y()) * INV_PI);
    uv -= uv.array().floor().matrix();
    Float pdf = m_cube_distrb.pdf(uv) /
                sqrt(std::max(square(dir.x()) + square(dir.z()), square(Epsilon))) * (.5f / square(M_PI));
    return pdf * G;
}

Spectrum EnvironmentMap::sampleDirect(const Vector2 &_rnd, DirectSamplingRecord &dRec) const {
    // sample a direction
    Ray     ray;
    Vector2 rnd(_rnd);
    Float   pdf;
    ray.org                = dRec.ref;
    std::tie(ray.dir, pdf) = sampleDirection(rnd);
    Intersection its;

    // ray intersect the bounding mesh
    bool valid = shape_ptr->rayIntersect(ray, its);
    assert(valid);

    Vector dir  = its.p - dRec.ref;
    Float  dist = dir.norm();
    dir /= dist;

    dRec.tri_id      = its.triangle_id;
    dRec.barycentric = its.barycentric;
    dRec.p           = its.p;
    dRec.dist        = dist;
    dRec.dir         = dir;
    dRec.n           = its.geoFrame.n;
    dRec.G           = std::abs(dRec.n.dot(-ray.dir)) / (dist * dist);
    dRec.uv          = its.uv;
    dRec.J           = 1.; // the bounding mesh won't change
    dRec.pdf         = detach(pdf * dRec.G);
    dRec.measure     = EMArea;
    dRec.emittance   = eval(dRec.n, -ray.dir); // REVIEW
    return dRec.emittance * dRec.G * dRec.J / dRec.pdf;
}

Spectrum EnvironmentMap::samplePosition(const Vector2 &, PositionSamplingRecord &) const {
    assert(false);
}

std::pair<Vector, Float> EnvironmentMap::sampleDirection(const Vector2 &_rnd) const {
    Vector2 rnd(_rnd);
    PSDR_ASSERT(m_ready);
    Float  pdf   = m_cube_distrb.sample_reuse(rnd);
    Float  theta = rnd.y() * M_PI, phi = rnd.x() * 2 * M_PI;
    Vector d = sphdir(theta, phi);
    d        = Vector(d.y(), d.z(), -d.x());

    Float inv_sin_theta = 1. /
                          sqrt(std::max(square(d.x()) + square(d.z()), square(Epsilon)));
    pdf *= inv_sin_theta * 0.5 / square(M_PI);

    d = psdr::transform_dir(m_to_world, d);
    return { d, pdf };
}

PSDR_IMPL_EMITTER_HELPER_FUNCTIONS(EnvironmentMap)