#include <render/integrator.h>
#include <render/scene.h>
#include <core/properties.h>

struct VolPrimaryEdgeIntegrator : IntegratorBoundary
{
    DiscreteDistribution edge_dist;
    std::vector<Vector2i> edge_indices; // [{face_id, edge_id}]
    VolPrimaryEdgeIntegrator(const Properties &props);
    VolPrimaryEdgeIntegrator(const Scene &scene);
    ArrayXd renderD(SceneAD &sceneAD,
                    RenderOptions &options, const ArrayXd &d_image) const;
    void configure(const Scene &scene) override;
};

struct VolDirectEdgeIntegrator : IntegratorBoundary
{
    DiscreteDistribution edge_dist;
    std::vector<Vector2i> edge_indices; // [{face_id, edge_id}]
    VolDirectEdgeIntegrator(const Properties &props);
    VolDirectEdgeIntegrator(const Scene &scene);
    ArrayXd renderD(SceneAD &sceneAD,
                    RenderOptions &options, const ArrayXd &d_image) const;
    void configure(const Scene &scene) override;
};

struct VolIndirectEdgeIntegrator : IntegratorBoundary
{
    DiscreteDistribution edge_dist;
    std::vector<Vector2i> edge_indices; // [{face_id, edge_id}]
    VolIndirectEdgeIntegrator(const Properties &props);
    VolIndirectEdgeIntegrator(const Scene &scene);
    ArrayXd renderD(SceneAD &sceneAD,
                    RenderOptions &options, const ArrayXd &d_image) const;
    void configure(const Scene &scene) override;
};

struct VolBoundaryIntegrator : IntegratorBoundary
{
    VolPrimaryEdgeIntegrator p;
    VolDirectEdgeIntegrator d;
    VolIndirectEdgeIntegrator i;
    VolBoundaryIntegrator(const Properties &props) : p(props), d(props), i(props) {
        auto scene = dynamic_cast<const Scene *>(props.get<Object*>("scene"));
        if (!scene)
            throw std::runtime_error("VolPrimaryEdgeIntegrator: No scene was specified!");
        configure(*scene);
    }

    VolBoundaryIntegrator(const Scene &scene) : p(scene), d(scene), i(scene) {}
    void configure(const Scene &scene) override
    {
        p.configure(scene);
        d.configure(scene);
        i.configure(scene);
    }
    void configure_primary(const Scene &scene)
    {
        p.configure(scene);
    }
    
    ArrayXd renderD(SceneAD &sceneAD,
                    RenderOptions &options, const ArrayXd &d_image) const
    {
        return p.renderD(sceneAD, options, d_image) +
               d.renderD(sceneAD, options, d_image) +
               i.renderD(sceneAD, options, d_image);
    }
};