#include <core/utils.h>
#include <core/math_func.h>
#include <Eigen/Geometry>
#include <iostream>
#include <core/logger.h>

Vector3 xfm_point(const Matrix4x4 &T, const Vector3 &pos)
{
    Vector4 pos_homo(pos.x(), pos.y(), pos.z(), 1.0f);
    pos_homo = T * pos_homo;
    return pos_homo.head(3) / pos_homo.w();
}

Vector3 xfm_vector(const Matrix4x4 &T, const Vector3 &vec)
{
    Matrix3x3 TT = T.block(0, 0, 3, 3);
    return TT * vec;
}

Vector2 squareToUniformDiskConcentric(const Vector2 &sample)
{
    Float r1 = 2.0f * sample.x() - 1.0f;
    Float r2 = 2.0f * sample.y() - 1.0f;
    Float phi, r;
    if (r1 == 0 && r2 == 0)
    {
        r = phi = 0;
    }
    else if (r1 * r1 > r2 * r2)
    {
        r = r1;
        phi = (M_PI / 4.0f) * (r2 / r1);
    }
    else
    {
        r = r2;
        phi = (M_PI / 2.0f) - (r1 / r2) * (M_PI / 4.0f);
    }
    Float sinPhi = sin(phi), cosPhi = cos(phi);
    // math::sincos(phi, sinPhi, cosPhi);
    return Vector2(r * cosPhi, r * sinPhi);
}

Vector2 invSquareToUniformDiskConcentric(const Vector2 &p)
{
    Float phi = atan2(p.y(), p.x());
    Float r = sqrt(pow(p.x(), 2) + pow(p.y(), 2));
    if (phi < -(M_PI / 4.0f)) {
        phi = phi + M_PI;
        r = -r;
    } else if (phi > M_PI * 3.0 / 4.0) {
        phi = phi - M_PI;
        r = -r;
    }
    Float r1;
    Float r2;
    if (phi > M_PI / 4.0f) {
        r2 = r;
        r1 = (M_PI / 2.0f - phi) / (M_PI / 4.0f) * r2;
    } else {
        r1 = r;
        r2 = phi / (M_PI / 4.0f) * r1;
    }
    return Vector2((r1 + 1.0f) / 2.0f, (r2 + 1.0f) / 2.0f);
}

Vector squareToCosineHemisphere(const Vector2 &sample)
{
    Vector2 p = squareToUniformDiskConcentric(sample);
    Float z = std::sqrt(
        std::max(static_cast<Float>(0.0f), 1.0f - p.x() * p.x() - p.y() * p.y()));
    return Vector(p.x(), p.y(), z);
}

Vector2 invSquareToCosineHemisphere(const Vector &dir)
{
    Vector2 p = Vector2(dir.x(), dir.y());
    return invSquareToUniformDiskConcentric(p);
}
Vector squareToCosineSphere(const Vector2 &_sample)
{
    Vector2 sample(_sample);
    bool flip;
    if (sample[0] < 0.5f)
    {
        flip = false;
        sample[0] *= 2.0f;
    }
    else
    {
        flip = true;
        sample[0] = 2.0f * (sample[0] - 0.5f);
    }
    Vector ret = squareToCosineHemisphere(sample);
    if (flip)
        ret[2] = -ret[2];
    return ret;
}

Vector squareToUniformSphere(const Vector2 &sample)
{
    Float z = 1.0f - 2.0f * sample.y();
    Float r = std::sqrt(std::max(static_cast<Float>(0.0f), 1.0f - z * z));
    Float sinPhi, cosPhi;
    sinPhi = sin(2.0f * M_PI * sample.x());
    cosPhi = cos(2.0f * M_PI * sample.x());
    // math::sincos(2.0f * M_PI * sample.x(), sinPhi, cosPhi);
    return Vector(r * cosPhi, r * sinPhi, z);
}

Vector2 invSquareToUniformSphere(const Vector &dir)
{
    Float z = dir.z();
    Float r = std::sqrt(std::max(static_cast<Float>(0.0f), 1.0f - z * z));
    Float cosPhi = dir.x() / r;
    Float sinPhi = dir.y() / r;
    Float phi = std::atan2(sinPhi, cosPhi);
    if (phi < 0.0f)
        phi += 2.0f * M_PI;
    return Vector2(phi / (2.0f * M_PI), 0.5f * (1.0f + z));
}

Float squareToCosineHemispherePdf(const Vector &d) { return INV_PI * d.z(); }

Float squareToCosineSpherePdf(const Vector &d)
{
    return 0.5f * INV_PI * std::abs(d.z());
}

Vector2 squareToGaussianDisk(const Vector2 &sample, Float sigma = 0.1) {Vector2 ret;
    Float rad = sqrt(-2.0 * log(1 - sample(0)));
	ret[0] = sigma * sqrt(-2.0 * log(1 - sample(0))) * cos(2 * M_PI * sample(1));
	ret[1] = sigma * sqrt(-2.0 * log(1 - sample(0))) * sin(2 * M_PI * sample(1));
    return ret;
}

Float squareToGaussianDiskPdf(const Vector2 &d, Float sigma = 0.1)
{
    // 1.0 / (sigma * sigma * 2.0 * M_PI) * exp(-0.5 * d.squaredNorm() / (sigma * sigma));
    return INV_PI * exp(-0.5 * d.squaredNorm() / (sigma * sigma)) / (sigma * sigma * 2.0);
}

void coordinateSystem(const Vector &_n, Vector &s, Vector &t)
{
    Matrix3x3 randRot(
        Eigen::AngleAxis<Float>(0.1f, Vector(0.1f, 0.2f, 0.3f).normalized())
            .toRotationMatrix());
    Matrix3x3 randRotInv = randRot.transpose();
    Vector n = randRot * _n;
    if (std::abs(n.x()) > std::abs(n.y()))
    {
        Float invLen = 1.0f / std::sqrt(n.x() * n.x() + n.z() * n.z());
        t = Vector(n.z() * invLen, 0.0f, -n.x() * invLen);
    }
    else
    {
        Float invLen = 1.0f / std::sqrt(n.y() * n.y() + n.z() * n.z());
        t = Vector(0.0, n.z() * invLen, -n.y() * invLen);
    }
    s = t.cross(n);
    s = randRotInv * s;
    t = randRotInv * t;
}

// void coordinateSystem(const Vector &a, Vector &b, Vector &c) {
//     if (std::abs(a.x()) > std::abs(a.y())) {
//         Float invLen = 1.0f / std::sqrt(a.x() * a.x() + a.z() * a.z());
//         c = Vector(a.z() * invLen, 0.0f, -a.x() * invLen);
//     } else {
//         Float invLen = 1.0f / std::sqrt(a.y() * a.y() + a.z() * a.z());
//         c = Vector(0.0f, a.z() * invLen, -a.y() * invLen);
//     }
//     b = c.cross(a);
// }

Float luminance(const Vector &v)
{
    return 0.212671f * v(0) + 0.715160f * v(1) + 0.072169f * v(2);
}

// bool isPixelValueValid(const Spectrum3f &val)
// {
//     for (int i = 0; i < 3; i++)
//     {
//         if (val(i) < -1e-6 || std::isinf(val(i)) || std::isnan(val(i)))
//         {
//             std::cout << "[invalid pixel value] " << val << std::endl;
//             return false;
//         }
//     }
//     return true;
// }

void progressIndicator(Float progress, int barWidth)
{
    printf("[");
    int pos = barWidth * progress;
    for (int i = 0; i < barWidth; ++i)
    {
        if (i < pos)
            printf("=");
        else if (i == pos)
            printf(">");
        else
            printf(" ");
    }
    printf("] %.0lf %%\r", progress * 100.);
    fflush(stdout);
}

Float fresnelDielectricExt(Float cosThetaI_, Float &cosThetaT_, Float eta)
{
    if (std::abs(eta - 1.0f) < Epsilon)
    {
        cosThetaT_ = -cosThetaI_;
        return 0.0f;
    }
    /* Using Snell's law, calculate the squared sine of the
     angle between the normal and the transmitted ray */
    Float scale = (cosThetaI_ > 0.0f) ? 1.0f / eta : eta,
          cosThetaTSqr =
              1.0f - (1.0f - cosThetaI_ * cosThetaI_) * (scale * scale);

    /* Check for total internal reflection */
    if (cosThetaTSqr < Epsilon)
    {
        cosThetaT_ = 0.0f;
        return 1.0f;
    }

    /* Find the absolute cosines of the incident/transmitted rays */
    Float cosThetaI = std::abs(cosThetaI_);
    Float cosThetaT = std::sqrt(cosThetaTSqr);

    Float Rs = (cosThetaI - eta * cosThetaT) / (cosThetaI + eta * cosThetaT);
    Float Rp = (eta * cosThetaI - cosThetaT) / (eta * cosThetaI + cosThetaT);

    cosThetaT_ = (cosThetaI_ > 0.0f) ? -cosThetaT : cosThetaT;

    /* No polarization -- return the unpolarized reflectance */
    return 0.5f * (Rs * Rs + Rp * Rp);
}

Vector refract(const Vector &wi, const Vector &n, Float eta, Float cosThetaT)
{
    if (cosThetaT < 0)
        eta = 1.0f / eta;
    return n * (wi.dot(n) * eta + cosThetaT) - wi * eta;
}

Vector refract(const Vector &wi, const Vector &n, Float eta)
{
    assert(std::abs(eta - 1.0) > Epsilon);

    Float cosThetaI = wi.dot(n);
    if (cosThetaI > 0)
        eta = 1.0f / eta;

    /* Using Snell's law, calculate the squared sine of the
     angle between the normal and the transmitted ray */
    Float cosThetaTSqr = 1.0f - (1.0f - cosThetaI * cosThetaI) * (eta * eta);

    /* Check for total internal reflection */
    if (cosThetaTSqr < Epsilon)
        return Vector::Zero();

    return n * (cosThetaI * eta -
                math::signum(cosThetaI) * std::sqrt(cosThetaTSqr)) -
           wi * eta;
}

Vector refract(const Vector &wi, const Vector &n, Float eta, Float &cosThetaT,
               Float &F)
{
    Float cosThetaI = wi.dot(n);
    F = fresnelDielectricExt(cosThetaI, cosThetaT, eta);

    if (std::abs(F - 1.0) < Epsilon) /* Total internal reflection */
        return Vector::Zero();

    if (cosThetaT < 0)
        eta = 1.0f / eta;
    return n * (eta * cosThetaI + cosThetaT) - wi * eta;
}

Array rayIntersectTriangle(const Vector &v0, const Vector &v1, const Vector &v2,
                           const Ray &ray)
{
    const Vector e1 = v1 - v0, e2 = v2 - v0;
    const Vector pvec = ray.dir.cross(e2);
    Float divisor = pvec.dot(e1);
    // Hack
    if (std::abs(divisor) < Epsilon)
        divisor = (divisor > 0) ? Epsilon : -Epsilon;
    const Vector s = ray.org - v0;
    const Float dot_s_pvec = s.dot(pvec);
    const Float u = dot_s_pvec / divisor;
    const Vector qvec = s.cross(e1);
    const Float dot_dir_qvec = ray.dir.dot(qvec);
    const Float v = dot_dir_qvec / divisor;
    const Float dot_e2_qvec = e2.dot(qvec);
    const Float t = dot_e2_qvec / divisor;
    return Vector(u, v, t);
}

bool rayIntersectTriangle(const Vector &v0, const Vector &v1, const Vector &v2,
                          const Ray &ray,
                          Float &u, Float &v, Float &t) {
    const Vector e1 = v1 - v0, e2 = v2 - v0;
    const Vector pvec = ray.dir.cross(e2);

    // lie in plane of triangle
    Float det = e1.dot(pvec);
    if (det > -1e-8f && det < 1e-8f)
        return false;

    Float divisor = pvec.dot(e1);
    // Hack
    if (std::abs(divisor) < Epsilon)
        divisor = (divisor > 0) ? Epsilon : -Epsilon;
    const Vector s          = ray.org - v0;
    const Float  dot_s_pvec = s.dot(pvec);
    u                       = dot_s_pvec / divisor;

    if (u < 0.0 || u > 1.0)
        return false;

    const Vector qvec         = s.cross(e1);
    const Float  dot_dir_qvec = ray.dir.dot(qvec);
    v                         = dot_dir_qvec / divisor;

    if (v < 0.0 || u + v > 1.0)
        return false;

    const Float dot_e2_qvec = e2.dot(qvec);
    t                       = dot_e2_qvec / divisor;

    return t >= ray.tmin && t <= ray.tmax;
}

Vector rayIntersectTriangleAD(const Vector &v0, const Vector &v1, const Vector &v2, const Ray &ray)
{
    const Vector e1 = v1 - v0, e2 = v2 - v0;
    const Vector pvec = ray.dir.cross(e2);
    Float divisor = pvec.dot(e1);
    // Hack
    if (std::abs(divisor) < Epsilon)
        divisor = (divisor > 0) ? Epsilon : -Epsilon;
    const Vector s = ray.org - v0;
    const Float dot_s_pvec = s.dot(pvec);
    const Float u = dot_s_pvec / divisor;
    const Vector qvec = s.cross(e1);
    const Float dot_dir_qvec = ray.dir.dot(qvec);
    const Float v = dot_dir_qvec / divisor;
    return (1.0f - u - v) * v0 + u * v1 + v * v2;
}
// INACTIVE_FN(rayIntersectTriangle, rayIntersectTriangle);

Float computeIntersectionInTri(const Vector &a, const Vector &b0,
                               const Vector &c0, const Vector &b1,
                               const Vector &c1, Float t0)
{
    // b0 = a + coeff_b * (b1-a) + 0 * (c1-a)
    Float coeff_b = (b0 - a).norm() / (b1 - a).norm();
    // c0 = a +    0 * (b1-a)    + coeff_c * (c1-a)
    Float coeff_c = (c0 - a).norm() / (c1 - a).norm();
    // p0 = b0 + t0 * (c0 - b0)
    Vector2 barycentric((1 - t0) * coeff_b, t0 * coeff_c);
    barycentric /= (barycentric.x() + barycentric.y());
    // p = a + bary.x * (b1-a) + bary.y * (c1-a)
    return barycentric.y();
}

Vector squareToEdgeRayDirection(const Vector2 &sample, const Vector &n0,
                                const Vector &n1, Float &pdf, bool avoid_inface)
{
    Float tmp = n0.dot(n1);
    // PSDR_WARN(tmp > -1.0f + EdgeEpsilon && tmp < 1.0f - EdgeEpsilon);
    Float phi0 = std::acos(tmp);
    pdf = 1.0f / (4.0f * phi0);

    Vector Z = (n0 + n1).normalized();
    Vector Y = n0.cross(Z).normalized();
    Vector X = Y.cross(Z);

    Float phi = (sample[0] - 0.5f) * phi0;

    // Avoid sampling directions that are almost within either plane
    if (avoid_inface)
        phi = clamp(phi, -0.5f * phi0 + EdgeEpsilon, 0.5f * phi0 - EdgeEpsilon);
    else 
        phi = clamp(phi, -0.5f * phi0, 0.5f * phi0);

    Vector X1 = X * std::cos(phi) + Z * std::sin(phi);

    Float a, b;
    if (sample[1] > 0.5f)
    {
        b = 4.0f * sample[1] - 3.0f;
        a = -math::safeSqrt(1.0f - b * b);
    }
    else
    {
        b = 4.0f * sample[1] - 1.0f;
        a = math::safeSqrt(1.0f - b * b);
    }

    Vector ret = X1 * a + Y * b;
    // assert(math::signum(ret.dot(n0))*math::signum(ret.dot(n1)) < -0.5f);
    return ret;
}

Vector dsquareToDEdgeRayDirection(const Vector2 &sample, const Vector2 &d_sample, const Vector &n0,
                                const Vector &n1)
{
    Float tmp = n0.dot(n1);
    // PSDR_WARN(tmp > -1.0f + EdgeEpsilon && tmp < 1.0f - EdgeEpsilon);
    Float phi0 = std::acos(tmp);

    Vector Z = (n0 + n1).normalized();
    Vector Y = n0.cross(Z).normalized();
    Vector X = Y.cross(Z);

    Float phi = (sample[0] - 0.5f) * phi0;
    Float d_phi = d_sample[0] * phi0;

    // Avoid sampling directions that are almost within either plane
    // phi = clamp(phi, -0.5f * phi0, 0.5f * phi0);

    Vector X1 = X * std::cos(phi) + Z * std::sin(phi);
    Vector d_X1 = -X * std::sin(phi) * d_phi + Z * std::cos(phi) * d_phi;
    // PSDR_INFO("d_X1: {}, {}, {}", d_X1[0], d_X1[1], d_X1[2]);
    Float a, b;
    Float d_a, d_b;
    if (sample[1] > 0.5f)
    {
        b = 4.0f * sample[1] - 3.0f;
        a = -math::safeSqrt(1.0f - b * b);
        d_b = 4.0f * d_sample[1];
        d_a = -d_b * b / a;
    }
    else
    {
        b = 4.0f * sample[1] - 1.0f;
        a = math::safeSqrt(1.0f - b * b);
        d_b = 4.0f * d_sample[1];
        d_a = -d_b * b / a;
    }
    // PSDR_INFO("d_a: {}, d_b: {}", d_a, d_b);
    // PSDR_INFO("d_X1: {}, {}, {}", d_X1[0], d_X1[1], d_X1[2]);
    // PSDR_INFO("d_phi: {}", d_phi);

    Vector d_ret = d_X1 * a + X1 * d_a + Y * d_b;
    
    return d_ret;
}

Vector2 dedgeRayDirectionToDSquare(const Vector2 sample, const Vector &dir, const Vector &d_dir, const Vector &n0,
                                 const Vector &n1) { // dy/dt=df(x)/dx*dx/dt, y=f(x) x=f^-1(y) 
                                                     // dx/dt=df^-1(y)/dy*dy/dt=1/df(x)/dx*dy/dt
    Vector2 d_sample;
    Float tmp = n0.dot(n1);
    // PSDR_WARN(tmp > -1.0f + EdgeEpsilon && tmp < 1.0f - EdgeEpsilon);
    Float phi0 = std::acos(tmp);

    Vector Z = (n0 + n1).normalized();
    Vector Y = n0.cross(Z).normalized();
    Vector X = Y.cross(Z);

    Float phi = (sample[0] - 0.5f) * phi0;

    // Avoid sampling directions that are almost within either plane
    phi = clamp(phi, -0.5f * phi0, 0.5f * phi0);

    Vector X1 = X * std::cos(phi) + Z * std::sin(phi);

    Float a, b;
    if (sample[1] > 0.5f)
    {
        b = 4.0f * sample[1] - 3.0f;
        a = -math::safeSqrt(1.0f - b * b);
    }
    else
    {
        b = 4.0f * sample[1] - 1.0f;
        a = -math::safeSqrt(1.0f - b * b);
    }
    Vector ret = X1 * a + Y * b;

    Float d_b = d_dir.dot(Y);
    Float d_a;

    // if (sample[1] > 0.5f)
    // {
        d_a = -d_b * b / a;
        d_sample[1] = d_b / 4.0f;
    // }
    // else
    // {
    //     d_a = -d_b * b / a;
    //     d_sample[1] = d_b / 4.0f;
    // }
    // PSDR_INFO("d_a: {}, d_b: {}", d_a, d_b);
    Vector d_X1 = (d_dir - Y * d_b - X1 * d_a) / a;
    // PSDR_INFO("d_X1: {}, {}, {}", d_X1[0], d_X1[1], d_X1[2]);
    // Vector d_X1 = -X * std::sin(phi) * d_phi + Z * std::cos(phi) * d_phi;
    Vector d_vec = -X * std::sin(phi) + Z * std::cos(phi);
    Float d_phi = d_X1.dot(d_vec) / (d_vec.squaredNorm());
    // PSDR_INFO("d_phi: {}", d_phi);
    d_sample[0] = d_phi / phi0;
    return d_sample;
}

Vector2 edgeRayDirectionToSquare(const Vector &dir, const Vector &n0,
                                 const Vector &n1)
{
    Vector2 sample;
    Float tmp = n0.dot(n1);
    // PSDR_WARN(tmp > -1.0f + EdgeEpsilon && tmp < 1.0f - EdgeEpsilon);
    Float phi0 = std::acos(tmp);

    Vector Z = (n0 + n1).normalized();
    Vector Y = n0.cross(Z).normalized();
    Vector X = Y.cross(Z);

    Float b = dir.dot(Y);
    Float a = math::safeSqrt(1.0f - b * b);
    Vector remainder = (dir - Y * b) / a;
    if (remainder.dot(X) < 0){
        a = -a;
        sample[1] = (b + 3.0f) / 4.0f;
    } else {
        sample[1] = (b + 1.0f) / 4.0f;
    }
    Vector X1 = (dir - Y * b) / a;
    Float cosphi = X1.dot(X);
    Float sinphi = X1.dot(Z);
    Float phi = std::atan2(sinphi, cosphi);
    sample[0] = (phi + 0.5f * phi0) / phi0;
    // assert(sample[0] >= 0.0f && sample[0] <= 1.0f);
    clamp(sample[0], 0.0, 1.0);
    clamp(sample[1], 0.0, 1.0);
    return sample;
}

bool rayLineIntersection(const Vector &origin, const Vector &dir, const Vector &v0, const Vector &v1,
                         Vector &P, Float &u, Float &t) {
    Vector P1 = origin;
    Vector P2 = origin + dir;
    Vector P3 = v0;
    Vector P4 = v1;
  
    Vector P13 = P1 - P3;
    Vector P43 = P4 - P3;
    Vector P21 = P2 - P1;
    
    Float d1343 = P13.dot(P43);
    Float d4321 = P43.dot(P21);
    Float d1321 = P13.dot(P21);
    Float d4343 = P43.dot(P43);
    Float d2121 = P21.dot(P21);

    Float denom = d2121 * d4343 - d4321 * d4321;
    if (abs(denom) == 0.0f) {
        PSDR_INFO("parallel line: ");
        PSDR_INFO("origin: {}, {}, {}", origin[0], origin[1], origin[2]);
        PSDR_INFO("dir: {}, {}, {}", dir[0], dir[1], dir[2]);
        Vector line_dir_norm = (v1 - v0).normalized();
        PSDR_INFO("v0: {}, {}, {}", v0[0], v0[1], v0[2]);
        PSDR_INFO("v1: {}, {}, {}", v1[0], v1[1], v1[2]);
        PSDR_INFO("line_dir: {}, {}, {}", line_dir_norm[0], line_dir_norm[1], line_dir_norm[2]);
        // assert(false); // parallel
        return false;
    }
        
    Float numer = d1343 * d4321 - d1321 * d4343;

    t = numer / denom;
    u = (d1343 + d4321 * t) / d4343;
    if (u < 0 || u > 1) 
        return false;

    P = P1 + t * P21;

    Vector sancheck = P3 + u * P43; 
    if ((P - sancheck).norm() > 1e-6) {
        return false;// same plane assertion
    }

    return true;
}

// Convert std::vector<Spectrum> to ArrayXd
ArrayXd from_spectrum_list_to_tensor(const std::vector<Spectrum> &spec_list, int n_pixels)
{
    ArrayXd arr = ArrayXd::Zero(n_pixels * 3);

    for (int i = 0; i < n_pixels; ++i)
    {
        Eigen::Array3d s = spec_list[i].array();
        arr[3 * i + 0] = s[0];
        arr[3 * i + 1] = s[1];
        arr[3 * i + 2] = s[2];
    }

    return arr;
}

// template <class T>
// void from_spectrum_list_to_ptr(const std::vector<T>& spec_list, int n_pixels,
//     ptr<float> p_image) {
//   for (int i = 0; i < n_pixels; ++i) {
//     p_image[3*i] = spec_list[i][0](0);
//     p_image[3*i+1] = spec_list[i][1](0);
//     p_image[3*i+2] = spec_list[i][2](0);
//   }
// }

// Convert ArrayXD to std::vector<Spectrum>
std::vector<Spectrum> from_tensor_to_spectrum_list(const ArrayXd &arr, int n_pixels)
{
    std::vector<Spectrum> spec_list;
    spec_list.reserve(n_pixels);

    for (int i = 0; i < n_pixels; ++i)
    {
        spec_list.push_back(Spectrum(arr(3 * i), arr(3 * i + 1), arr(3 * i + 2)));
    }

    return spec_list;
}

std::vector<Spectrum> from_tensor_to_spectrum_list_scale(const ArrayXd &arr, const Spectrum& scale, int n_pixels)
{
    std::vector<Spectrum> spec_list;
    spec_list.reserve(n_pixels);

    for (int i = 0; i < n_pixels; ++i)
    {
        spec_list.push_back(Spectrum(arr(3 * i), arr(3 * i + 1), arr(3 * i + 2)) * scale);
    }

    return spec_list;
}

// INACTIVE_FN(squareToCosineHemisphere, squareToCosineHemisphere);
// INACTIVE_FN(squareToCosineHemispherePdf, squareToCosineHemispherePdf);
INACTIVE_FN(detach_Float, detach<Float>);
INACTIVE_FN(detach_Vector2, detach<Vector2>);
INACTIVE_FN(detach_Vector, detach<Vector>);
INACTIVE_FN(detach_Matrix, detach<Matrix3x3>);
