/*
 * SPDX-FileCopyrightText: Copyright (c) 2022 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
 * SPDX-License-Identifier: BSD-3-Clause
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice, this
 * list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. Neither the name of the copyright holder nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "Diffuse.h"

namespace EDX
{
    namespace TensorRay
    {
        Expr Diffuse::EvalInner(const Intersection& isect, const Expr& wo, const Expr& wi, TransportMode mode) const
        {
            auto reflect = BSDFCoords::CosTheta(wo) > Zeros(1) && BSDFCoords::SameHemisphere(wo, wi);
            return Scalar(float(Math::EDX_INV_PI)) * reflect;
        }

        Expr Diffuse::OutDirValid(const Intersection& isect, const Expr& wo) const
        {
            auto dotShNorm = VectorDot(isect.mNormal, wo);
            auto dotGeoNorm = VectorDot(isect.mGeoNormal, wo);
            return dotShNorm > Scalar(EDGE_EPSILON) && dotGeoNorm > Scalar(EDGE_EPSILON);
        }

        Expr Diffuse::SampleInner(const Intersection& isect, const Expr& wo, const Expr& samples, Expr* pWi, Expr* pPdf) const
        {
            TENSORRAY_ASSERT_MSG(false, std::string("Diffuse::SampleInner() not implemented!"));
            return Expr();
        }

        void Diffuse::SampleOnly(const Intersection& isect, const Expr& wo, const Expr& samples, Expr* pWi, Expr* pPdf) const
        {
#if USE_PROFILING
            nvtxRangePushA(__FUNCTION__);
#endif
            Expr localWo = isect.WorldToLocal(wo);
            Expr localWi = Sampling::CosineSampleHemisphere(samples);
            *pWi = isect.LocalToWorld(localWi);
            *pPdf = PdfInner(isect, localWo, localWi);
#if USE_PROFILING
            nvtxRangePop();
#endif
        }

        Expr Diffuse::PdfInner(const Intersection& isect, const Expr& wo, const Expr& wi) const
        {
            auto reflect = BSDFCoords::CosTheta(wo) > Zeros(1) && BSDFCoords::SameHemisphere(wo, wi);

            return BSDFCoords::AbsCosTheta(wi) / Scalar(float(Math::EDX_PI)) * reflect;
        }
    }
}