/*
 * Created Date: Wednesday, September 21st 2022, 2:53:15 pm
 * Author: Yu-Chen Wang
 * 
 * Copyright (c) 2022 yucwang
 */

#pragma once
#define TINYEXR_IMPLEMENTATION
// Use the MINIZ hosted by TensorRay rather than from System
#define TINYEXR_USE_MINIZ 0
#include "miniz.h"
#include "tinyexr.h"

#include "../Tensor/Tensor.h"

namespace EDX {
namespace TensorRay
{

struct ImageLoadRecord {
    int width;
    int height;
    bool success = false;
    Tensorf image;
};

ImageLoadRecord LoadEXRFromFile(const char* exr_path) {
    float* fPtr;
    int width, height;
    const char* err = NULL;

    ImageLoadRecord record;
    int res = LoadEXR(&fPtr, &width, &height, exr_path, &err);
    if (res != TINYEXR_SUCCESS) {
        if (err) {
            fprintf(stderr, "Load OpenEXR Failed: %s\n", err);
            FreeEXRErrorMessage(err);
        }
    } else {
        int numChannels = 3;
        std::vector<float> transformedImage(width * height * numChannels);
        for (int i = 0; i < height * width; ++i) {
            for (int j = 0; j < numChannels; ++j) {
                transformedImage[j * height * width + i] = fPtr[4 * i + j];
            }
        }
        record.image.Assign(&transformedImage[0], { numChannels, height * width });
        record.height = height;
        record.width = width;
        record.success = true;
        free(fPtr);
    }

    return record;
}

bool SaveEXR(const float* rgb, int width, int height, const char* outfilename, bool isTransposed) 
{
    EXRHeader header;
    InitEXRHeader(&header);

    EXRImage image;
    InitEXRImage(&image);

    image.num_channels = 3;

    std::vector<float> images[3];
    images[0].resize(width * height);
    images[1].resize(width * height);
    images[2].resize(width * height);

    if (!isTransposed)
    {
        // Split RGBRGBRGB... into R, G and B layer
        for (int i = 0; i < width * height; i++) {
            images[0][i] = rgb[3 * i + 0];
            images[1][i] = rgb[3 * i + 1];
            images[2][i] = rgb[3 * i + 2];
        }
    }
    else
    {
        // Split R..RG..GB..B into R, G and B layer
        for (int i = 0; i < width * height; i++) {
            images[0][i] = rgb[i];
            images[1][i] = rgb[i + width * height];
            images[2][i] = rgb[i + width * height * 2];
        }
    }

    float* image_ptr[3];
    image_ptr[0] = &(images[2].at(0)); // B
    image_ptr[1] = &(images[1].at(0)); // G
    image_ptr[2] = &(images[0].at(0)); // R

    image.images = (unsigned char**)image_ptr;
    image.width = width;
    image.height = height;

    header.num_channels = 3;
    header.channels = (EXRChannelInfo*)malloc(sizeof(EXRChannelInfo) * header.num_channels);
    // Must be (A)BGR order, since most of EXR viewers expect this channel order.
    strncpy_s(header.channels[0].name, "B", 255); header.channels[0].name[strlen("B")] = '\0';
    strncpy_s(header.channels[1].name, "G", 255); header.channels[1].name[strlen("G")] = '\0';
    strncpy_s(header.channels[2].name, "R", 255); header.channels[2].name[strlen("R")] = '\0';

    header.pixel_types = (int*)malloc(sizeof(int) * header.num_channels);
    header.requested_pixel_types = (int*)malloc(sizeof(int) * header.num_channels);
    for (int i = 0; i < header.num_channels; i++) {
        header.pixel_types[i] = TINYEXR_PIXELTYPE_FLOAT; // pixel type of input image
        header.requested_pixel_types[i] = TINYEXR_PIXELTYPE_HALF; // pixel type of output image to be stored in .EXR
    }

    const char* err = NULL; // or nullptr in C++11 or later.
    int ret = SaveEXRImageToFile(&image, &header, outfilename, &err);
    if (ret != TINYEXR_SUCCESS) {
        fprintf(stderr, "Save EXR err: %s\n", err);
        FreeEXRErrorMessage(err); // free's buffer for an error message
        return ret;
    }

    free(header.channels);
    free(header.pixel_types);
    free(header.requested_pixel_types);

    return ret;
}

} // namespace TensorRay
} // EDX