/*
 * SPDX-FileCopyrightText: Copyright (c) 2022 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
 * SPDX-License-Identifier: BSD-3-Clause
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice, this
 * list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. Neither the name of the copyright holder nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#pragma once
#include "RoughDielectric.h"

namespace EDX
{
    namespace TensorRay
    {
        Expr RoughDielectric::EvalInner(const Intersection& isect, const Expr& wo, const Expr& wi, TransportMode mode) const
        {
            auto cosThetaO = BSDFCoords::CosTheta(wo);
            auto cosThetaI = BSDFCoords::CosTheta(wi);

            // Ignore perfectly grazing configurations
            auto active = (cosThetaO != Scalar(0.f) && cosThetaI != Scalar(0.f));

            // Determine the type of interaction        
            auto reflect = (cosThetaO * cosThetaI > Scalar(0.f));

            // Determine the relative index of refraction
            auto eta = Where(cosThetaO > Scalar(0.f), mEta, mInvEta);
            auto invEta = Where(cosThetaO > Scalar(0.f), mInvEta, mEta);

            // Compute the half-vector
            auto wh = VectorNormalize(wo + wi * Where(reflect, Scalar(1.f), eta));

            // Ensure that the half-vector points into the same hemisphere as the macrosurface normal
            wh = MulSign(wh, BSDFCoords::CosTheta(wh));
            
            auto roughness = mRoughness->Eval(isect.mTexcoord);
            auto D = GGX_D(wh, roughness);
            active = (active && D != Scalar(0.0f));

            auto G = GGX_G(wo, wi, wh, roughness);

            // Fresnel term
            auto coso = VectorDot(wo, wh);
            auto Fr = FresnelDielectric(mEta, coso);

            auto isEvalR = (active && reflect);
            auto isEvalT = (active && ~reflect);
            
            // Reflectance
            auto reflValue = Fr * D * G / (Scalar(4.0f) * Abs(cosThetaO) * Abs(cosThetaI));
            reflValue = reflValue * mSpecularReflectance->Eval(isect.mTexcoord);

            // Transmittance
            /* Missing term in the original paper: account for the solid angle
               compression when tracing radiance -- this is necessary for
               bidirectional methods. */
            auto scale = (mode == TransportMode::Radiance) ? Square(invEta) : Scalar(1.f);
            auto transValue = Abs(
                (scale * (Scalar(1.f) - Fr) * D * G * eta * eta * VectorDot(wo, wh) * VectorDot(wi, wh)) /
                (cosThetaO * Square(VectorDot(wo, wh) + eta * VectorDot(wi, wh)) * cosThetaI));
            transValue = transValue * mSpecularTransmittance->Eval(isect.mTexcoord);

            auto res = Where(isEvalR, reflValue, Where(isEvalT, transValue, Scalar(0.f)));
            return res;
        }

        Expr RoughDielectric::SampleInner(const Intersection& isect, const Expr& wo, const Expr& samples, Expr* pWi, Expr* pPdf) const
        {
            TENSORRAY_ASSERT_MSG(false, std::string("RoughDielectric::SampleInner() not implemented!"));
            return Expr();
        }

        void RoughDielectric::SampleOnly(const Intersection& isect, const Expr& wo, const Expr& samples, Expr* pWi, Expr* pPdf) const
        {
#if USE_PROFILING
            nvtxRangePushA(__FUNCTION__);
#endif
            auto roughness = mRoughness->Eval(isect.mTexcoord);
            
            Expr localWo = isect.WorldToLocal(wo);
            auto cosThetaO = BSDFCoords::CosTheta(localWo);

            auto active = (cosThetaO != Scalar(0.f));

            // Sample the microfacet normal
            auto pwo = MulSign(localWo, cosThetaO);
            auto wh = GGX_VNDF_Sample(pwo, roughness, samples);
            auto microfacetPdf = GGX_VNDF_Pdf(pwo, wh, roughness);
            active = (active && microfacetPdf != Scalar(0.f));

            Expr cosThetaT, etaIT, etaTI;
            auto Fr = FresnelDielectric(mEta, VectorDot(localWo, wh), &cosThetaT, &etaIT, &etaTI);

            auto isRefl = (active && Z(samples) <= Fr);
            auto isTrans = (active && Z(samples) > Fr);

            // Select the lobe to be sampled
            auto pdf = microfacetPdf * Where(isRefl, Fr, Where(isTrans, Scalar(1.f) - Fr, Scalar(0.f)));
            auto bsEta = Where(isRefl, Scalar(1.f), etaIT);

            // Sample directions
            auto localWi = Where(isRefl, VectorReflect(localWo, wh),
                Where(isTrans, VectorRefract(localWo, wh, cosThetaT, etaTI), Scalar(0.f)));
            auto dwh_dwi = Where(isRefl, Scalar(1.0f) / (Scalar(4.0f) * VectorDot(localWi, wh)),
                Where(isTrans, (Square(bsEta) * VectorDot(localWi, wh)) / Square(VectorDot(localWo, wh) + bsEta * VectorDot(localWi, wh)), Scalar(0.f)));

            pdf = pdf * Abs(dwh_dwi) * Smith_G(localWi, wh, roughness); // Smith_G might be unneeded.
            pdf = Where(pdf < Scalar(EPSILON), Scalar(0.f), pdf);

            *pPdf = pdf;
            *pWi = isect.LocalToWorld(Detach(localWi));

#if USE_PROFILING
            nvtxRangePop();
#endif
        }

        Expr RoughDielectric::PdfInner(const Intersection& isect, const Expr& wo, const Expr& wi) const
        {
            auto cosThetaO = BSDFCoords::CosTheta(wo);
            auto cosThetaI = BSDFCoords::CosTheta(wi);

            auto active = (cosThetaO != Scalar(0.f) && cosThetaI != Scalar(0.f));
            auto reflect = (cosThetaO * cosThetaI > Scalar(0.f));

            auto eta = Where(cosThetaO > Scalar(0.f), mEta, mInvEta);
            auto wh = VectorNormalize(wo + wi * Where(reflect, Scalar(1.f), eta));
            wh = MulSign(wh, BSDFCoords::CosTheta(wh));

            active = (active && VectorDot(wo, wh) * cosThetaO > Scalar(0.f) && VectorDot(wi, wh) * cosThetaI > Scalar(0.f));

            auto dwh_dwi = Where(reflect, Scalar(1.0f) / (Scalar(4.0f) * VectorDot(wi, wh)),
                eta * eta * VectorDot(wi, wh) / Square(VectorDot(wo, wh) + eta * VectorDot(wi, wh)));

            auto roughness = mRoughness->Eval(isect.mTexcoord);
            auto pwo = MulSign(wo, BSDFCoords::CosTheta(wo));
            auto prob = GGX_VNDF_Pdf(pwo, wh, roughness);

            auto Fr = FresnelDielectric(mEta, VectorDot(wo, wh));
            prob = prob * Where(reflect, Fr, Scalar(1.f) - Fr);// *Smith_G(wi, wh, roughness); // Smith_G might be unneeded.
            auto pdf = Where(active, prob * Abs(dwh_dwi), Scalar(0.f));
            return pdf;
        }

        void RoughDielectricTests()
        {
            Tensorf roughness({ 0.05f });
            roughness.SetRequiresGrad(true);

            RoughDielectric bsdf = RoughDielectric(new ConstantTexture(roughness),
                1.5f, 1.0f, 
                new ConstantTexture(Vector3(1.f, 1.f, 1.f)),
                new ConstantTexture(Vector3(1.f, 1.f, 1.f)));

            int N = 2;
            Tensorf sampleIn = Tensorf::RandomFloat(Shape({ N }, VecType::Vec2));
            Tensorf sampleOut = Tensorf::RandomFloat(Shape({ N }, VecType::Vec2));

            //Tensorf vIn = Sampling::CosineSampleHemisphere(sampleIn);
            //Tensorf vOut = Sampling::CosineSampleHemisphere(sampleOut);

            Tensorf vIn = Tensorf({ Vector3(1.f, 0.f, 1.f), Vector3(1.f, 0.f, 1.f) });
            Tensorf vOut = Tensorf({ Vector3(-1.f, 0.f, 1.f), Vector3(-0.5f, 0.f, -1.f) });

            Intersection isect;
            Tensorf temp = Tensorf({ Vector3(0.0f, 0.0f, 1.0f) }, false);
            Tensorf normal = Broadcast(temp, Shape({ N }, VecType::Vec3));
            isect.mNormal = normal;
            isect.mNormal = VectorNormalize(isect.mNormal);
            CoordinateSystem(isect.mNormal, &isect.mTangent, &isect.mBitangent);

            Tensorf texcoord = Tensorf::RandomFloat(Shape({ N }, VecType::Vec2));
            isect.mTexcoord = texcoord;
            Tensorf f = bsdf.Eval(isect, VectorNormalize(vIn), VectorNormalize(vOut)) * Abs(VectorDot(VectorNormalize(vOut), isect.mNormal));
            std::cout << "f: " << f << "\n";

            Tensorf pdf = bsdf.Pdf(isect, VectorNormalize(vIn), VectorNormalize(vOut));
            std::cout << "pdf: " << pdf << "\n";

            Tensorf samples = Tensorf({ Vector3(0.1f, 0.9f, 0.5f), Vector3(0.3f, 0.6f, 0.4f) });
            Expr pWi, pPdf;
            bsdf.SampleOnly(isect, VectorNormalize(vIn), samples, &pWi, &pPdf);
            Tensorf bsWi = pWi;
            Tensorf bsPdf = pPdf;

            std::cout << "bs_wi: " << bsWi << "\n";
            std::cout << "bs_pdf: " << bsPdf << "\n";

            f = bsdf.Eval(isect, VectorNormalize(vIn), bsWi) * Abs(VectorDot(bsWi, isect.mNormal));
            std::cout << "f: " << f << "\n";

            pdf = bsdf.Pdf(isect, VectorNormalize(vIn), bsWi);
            std::cout << "pdf: " << pdf << "\n";

            Tensorf weight = f / Detach(pdf);
            std::cout << "weight: " << f / pdf << "\n";

            ValidateForwardDiff(weight, bsdf.mRoughness->GetTextureTensor());

            ParameterPool::GetHandle().clear();
        }
    }
}