# Script to update legal headers across the code base.

import os
import sys
import re
import subprocess
from datetime import datetime
from glob import glob

# List of file extensions requiring the legal header.
CPP_EXTENSIONS = ['.h', '.c', '.hpp', '.cpp', '.cu', '.cuh']
PY_EXTENSIONS = ['.py']

EXCLUDE_CPP_FILENAMES = ['stb_image', 'pugi', 'miniz', 'tinyexr', 'jitify2']
EXCLUDE_PY_FILENAMES = ['__init__']

# Public legal header.
PUBLIC_CPP_HEADER_TEMPLATE = """
/*
 * SPDX-FileCopyrightText: Copyright (c) <year> NVIDIA CORPORATION & AFFILIATES. All rights reserved.
 * SPDX-License-Identifier: BSD-3-Clause
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice, this
 * list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. Neither the name of the copyright holder nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
"""

PUBLIC_PY_HEADER_TEMPLATE = """
# SPDX-FileCopyrightText: Copyright (c) <year> NVIDIA CORPORATION & AFFILIATES. All rights reserved.
# SPDX-License-Identifier: BSD-3-Clause
# 
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# 1. Redistributions of source code must retain the above copyright notice, this
# list of conditions and the following disclaimer.
#
# 2. Redistributions in binary form must reproduce the above copyright notice,
# this list of conditions and the following disclaimer in the documentation
# and/or other materials provided with the distribution.
#
# 3. Neither the name of the copyright holder nor the names of its
# contributors may be used to endorse or promote products derived from
# this software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
"""


DEFAULT_FIRST_YEAR=2022
DEFAULT_LAST_YEAR=2022


def get_sources(include_dirs, exclude_dirs, exclude_filenames, extensions):
    sources = []

    # get project root directory.
    project_root = os.path.abspath(os.path.join(os.path.split(sys.argv[0])[0], "../"))

    # add files from included directories
    for d in include_dirs:
        sources += glob(os.path.join(project_root, d), recursive=True)

    # remove files from excluded directories
    for d in exclude_dirs:
        sources = list(filter(lambda p: not p.startswith(os.path.abspath(os.path.join(project_root, d))), sources))

    # only keep files with valid extension
    extension_set = set(extensions)
    sources = list(filter(lambda p: os.path.splitext(p)[1] in extension_set, sources))

    res = []
    for fname in sources:
        to_exclude = False
        for ex_fname in exclude_filenames:
            if ex_fname in fname:
                to_exclude = True
                break

        if not to_exclude:
            res.append(fname)

    return res


def get_last_modify_year(path):
    cmd = ["git", "log", "-1", "--follow", "--pretty=%aI", path]
    result = subprocess.run(cmd, stdout=subprocess.PIPE)
    str = result.stdout.decode("utf-8")
    year = int(str[0:4])
    return year


def fix_legal_header(include_dirs, exclude_dirs, exclude_filenames, extensions, header_template):
    # regular expression matching the first block comment (after whitespace)
    rheader = re.compile(r"^\s*\/\*((\*(?!\/)|[^*])*)\*\/\s*")

    # regular expression matching the beginning of a comment
    rcomment = re.compile(r"^\s*\/[\*\/]")

    # prepare header text
    header_template = header_template.strip() + "\n"

    for p in get_sources(include_dirs, exclude_dirs, exclude_filenames, extensions):
        print("Processing %s" % (p))

        first_year = DEFAULT_FIRST_YEAR
        #last_year = max(DEFAULT_LAST_YEAR, get_last_modify_year(p))
        last_year = DEFAULT_LAST_YEAR

        #header = header_template.replace("{years}", "%d-%02d" % (first_year, last_year % 100))
        header = header_template.replace("<year>", "%d" % (last_year))

        # read file and remove existing header
        if p.endswith(".py"):
            fin = open(p)
            while True:
                first_line = fin.readline()
                if not (first_line.startswith("#") or first_line.startswith("\n")):
                    break
            text = first_line + fin.read()
        else:
            text = open(p).read()
            if rheader.match(text):
                text = rheader.sub("", text)

        # add extra newline when text starts with a comment
        if rcomment.match(text):
            text = "\n" + text

        # add new header
        text = header + "\n" + text

        text = text.replace("\xef\xbb\xbf", "")

        # write back file
        open(p, "w").write(text)


# fix public headers
fix_legal_header(
    include_dirs=["TensorRay.cpp", "TensorRay_debug.cpp", "LibTensorRay/**"],
    exclude_dirs=[],
    exclude_filenames=EXCLUDE_CPP_FILENAMES,
    extensions=CPP_EXTENSIONS,
    header_template=PUBLIC_CPP_HEADER_TEMPLATE
)

fix_legal_header(
    include_dirs=["pyTensorRay/**", "example/**"],
    exclude_dirs=["example/inverse_rendering/debug",
                  "example/inverse_rendering/scenes",
                  "example/validation_box_rfilter"],
    exclude_filenames=EXCLUDE_PY_FILENAMES,
    extensions=PY_EXTENSIONS,
    header_template=PUBLIC_PY_HEADER_TEMPLATE
)
