import argparse
import dataclasses
from typing import List

import numpy as np
from pypsdr.common import gen_cameras
from pypsdr.optimize import Camera, gen_camera_positions
import gin
from pypsdr.utils.io import imwrite, mkdir
from pypsdr.validate import RenderOptions, Scene
import psdr_cpu
import os
from pypsdr.render import render_save_multi_rotation


@gin.configurable
# generate the constructor
@dataclasses.dataclass
class PreprocessRunner:
    scene_file: str
    batch_size: int = 1
    radius: float = 1.0
    out_dir: str = "./"
    hemisphere: bool = False
    integrator: psdr_cpu.Integrator = psdr_cpu.Direct()
    camera_info: Camera = Camera()

    def run(self):
        psdr_cpu.set_verbose(True)
        self.scene = Scene(self.scene_file)
        self.width = self.scene.camera.width
        self.height = self.scene.camera.height
        self.fov = self.scene.camera.fov
        with gin.config_scope('tar'):
            self.options = RenderOptions(gin.REQUIRED)
        mkdir(self.out_dir)
        mkdir(os.path.join(self.out_dir, "target"))
        obj_pos = np.loadtxt('obj_pos.txt')
        render_save_multi_rotation(self.scene, obj_pos, self.options, self.integrator, os.path.join(self.out_dir, "target"))
        exit()

if __name__ == "__main__":
    default_config = './mirror.conf'
    parser = argparse.ArgumentParser(
        description='Script for generating validation results')
    parser.add_argument('config_file', metavar='config_file',
                        type=str, nargs='?', default=default_config, help='config file')
    args, unknown = parser.parse_known_args()
    # Dependency injection: Arguments are injected into the function from the gin config file.
    gin.parse_config_file(args.config_file, skip_unknown=True)
    runner = PreprocessRunner()
    runner.run()
