#pragma once
#ifndef INTEGRATOR_AD_PATHSPACE_H__
#define INTEGRATOR_AD_PATHSPACE_H__

#include "intersection.h"
#include "intersectionAD.h"
#include "integrator.h"
#include "phase.h"
#include <vector>
#include <omp.h>

#define BDPT_MAX_THREADS 256
#define BDPT_MAX_PATH_LENGTH 100

struct Ray;
struct RayAD;
struct Edge;
struct IntersectionAD;
struct Medium;

struct EventRecord
{
    Intersection its;
    Vector scatter;
    Vector wi;
    const Medium *ptr_med;
    bool onSurface;
    Vector x() const {
        return onSurface ? its.p : scatter;
    }
};

struct EventRecordAD
{
    IntersectionAD its;
    VectorAD scatter;
    VectorAD wi;
    const Medium *ptr_med;
    bool onSurface;
    EventRecordAD(){};
    // init a surface event
    EventRecordAD(const IntersectionAD &its) : its(its), onSurface(true){};
    // init a volume event
    EventRecordAD(const VectorAD &scatter, const VectorAD &wi, const Medium *ptr_med)
        : scatter(scatter), wi(wi), ptr_med(ptr_med), onSurface(false){};

    SpectrumAD fAD(const Scene &scene, const VectorAD &wi, const VectorAD &wo) const;                           // evaluate phase/bsdf function
    bool sample(const Scene &scene, RndSampler *sampler, const Vector &wi, Vector &wo, Float &pdf) const;       // sample a direction
    VectorAD &x() { return onSurface ? its.p : scatter; }                                                       // get position
    const Medium *getMedium(const Vector &dir) const { return onSurface ? its.getTargetMedium(dir) : ptr_med; } // get target medium
    SpectrumAD f(const Scene &scene, const VectorAD &wo) const;                                                 // evaluate bsdf/phase function
};

struct GuidingOptions {
    int type;
    std::vector<int> params;

    size_t num_cam_path;
    size_t num_light_path;
    float search_radius;            // Only necessary when using [Type 2] guiding
    bool quiet;

    GuidingOptions(int type, const std::vector<int> &params):
        type(type), params(params), num_cam_path(0), num_light_path(0), search_radius(0.0), quiet(false) {};
    GuidingOptions(int type, const std::vector<int> &params, size_t num_cam_path, size_t num_light_path):
        type(type), params(params),
        num_cam_path(num_cam_path), num_light_path(num_light_path), search_radius(0.0), quiet(false) {};
    GuidingOptions(int type, const std::vector<int> &params, size_t num_cam_path, size_t num_light_path, float search_radius):
        type(type), params(params),
        num_cam_path(num_cam_path), num_light_path(num_light_path), search_radius(search_radius), quiet(false) {};
};

struct RadImpNode {
    Vector p;
    Spectrum val;
    int depth;
};


struct IntegratorAD_PathSpace : Integrator {

    IntegratorAD_PathSpace();
    virtual ~IntegratorAD_PathSpace();

    virtual void render(const Scene &scene, const RenderOptions &options, ptr<float> rendered_image) const;

    /*******    Interior Term    ******/
    virtual void renderInterior(const Scene &scene, const RenderOptions &options, ptr<float> rendered_image) const = 0;

    /*******    Boundary Term    ******/
    virtual void renderEdgesPrimary(const Scene &scene, const RenderOptions &options, ptr<float> rendered_image) const;
    virtual void renderEdgesDirect(const Scene &scene, const RenderOptions &options, ptr<float> rendered_image) const;
    virtual void renderEdges(const Scene &scene, const RenderOptions &options, ptr<float> rendered_image) const;
    virtual void renderEdgesPointLight(const Scene &scene, const RenderOptions &options, ptr<float> rendered_image) const;  // Point light primary edge

    /*******    Preprocessing for importance sampling boundary term    ******/
    virtual void preprocess(const Scene &scene, int max_bounces, const GuidingOptions& opts, ptr<float> data) const;
    virtual void preprocessPrimary(const Scene &scene, int max_bounces, const GuidingOptions& opts, ptr<float> data) const;
    virtual void preprocessDirect(const Scene &scene, int max_bounces, const GuidingOptions& opts, ptr<float> data) const;
    virtual void preprocessIndirect(const Scene &scene, int max_bounces, const GuidingOptions& opts, ptr<float> data) const;
    virtual void preprocessDirectPointLight(const Scene &scene, int max_bounces, const GuidingOptions& opt, ptr<float> data) const;

    /******    Helper functions    ****/
    virtual std::string getName() const = 0;

    mutable omp_lock_t messageLock;
};

#endif //INTEGRATOR_AD_PATHSPACE_H__
